import datetime
import logging

from ..util.parsing import StreamParser
from .base import WialonCombinePacket


LOGGER = logging.getLogger(__name__)


class WialonCombineDataPacket(WialonCombinePacket):

    __slots__ = ['records']

    def __init__(self, records):
        self.records = records

    @classmethod
    def from_generic_packet(cls, generic_packet):
        parser = WialonCombineDataPacketParser()
        return parser.parse(generic_packet.data)

    @classmethod
    def get_type(cls):
        return 'data'

    def to_dict(self):
        return {
            'records': [r.to_dict() for r in self.records],
        }


class WialonCombineDataPacketRecord(object):

    __slots__ = ['timestamp', 'subrecords']

    def __init__(self, timestamp, subrecords):
        self.timestamp = timestamp
        self.subrecords = subrecords

    @property
    def date(self):
        return (
            datetime.datetime.utcfromtimestamp(self.timestamp)
            .replace(tzinfo=datetime.timezone.utc)
        )

    def to_dict(self):
        return {
            'type': 'record',
            'timestamp': self.timestamp,
            'subrecords': [sr.to_dict() for sr in self.subrecords],
        }


class WialonCombineDataPacketParser(object):

    def parse(self, raw_data):
        records = []
        sp = StreamParser(iter(raw_data))

        while True:
            try:
                timestamp = sp.parse_int()
            except StopIteration:
                break

            count = sp.parse_byte()

            subrecords = []
            for _ in range(count):
                subrecord = self._parse_subrecord(sp)
                LOGGER.info(subrecord.to_dict())
                subrecords.append(subrecord)

            record = WialonCombineDataPacketRecord(
                timestamp=timestamp,
                subrecords=subrecords,
            )
            records.append(record)

        packet = WialonCombineDataPacket(records=records)

        return packet

    def _parse_subrecord(self, sp):
        type_ = sp.parse_extensible_byte()
        if type_ == 0:
            subrecord = self._parse_custom_parameters(sp)
        elif type_ == 1:
            subrecord = self._parse_position_data(sp)
        elif type_ == 12:
            subrecord = self._parse_driver_message(sp)
        else:
            raise RuntimeError('Unknown subrecord type: {}'.format(type_))
        return subrecord

    def _parse_custom_parameters(self, sp):
        params = {}
        count = sp.parse_extensible_byte()
        for _ in range(count):
            sensor_number, sensor_value = self._parse_custom_parameter(sp)
            params[sensor_number] = sensor_value
        custom_parameters = WialonCombineDataPacketCustomParameters(params=params)
        return custom_parameters

    def _parse_custom_parameter(self, sp):
        sensor_number = sp.parse_extensible_byte()
        sensor_value = self._parse_custom_parameter_sensor_value(sp)
        return sensor_number, sensor_value

    def _parse_custom_parameter_sensor_value(self, sp):
        sensor_type = sp.parse_byte()
        data_type = sensor_type & 0b11111
        power = (sensor_type & 0b111 << 5) >> 5

        if data_type < 10:
            nbytes, fmt = {
                0: (1, 'B'),
                1: (2, 'H'),
                2: (4, 'I'),
                3: (8, 'Q'),
                4: (1, 'b'),
                5: (4, 'i'),
                6: (2, 'h'),
                7: (8, 'q'),
                8: (4, 'f'),
                9: (8, 'd'),
            }[data_type]
            value = sp.parse(nbytes, fmt)
        elif data_type == 10:
            value = sp.parse_string()
        else:
            raise RuntimeError('Unknown data type: %s', data_type)

        if power > 0:
            value /= 10 ** power

        return value

    def _parse_position_data(self, sp):
        lat = sp.parse_int() / 1000000
        lon = sp.parse_int() / 1000000
        speed = sp.parse_short()   # km/h
        course = sp.parse_short()  # [0, 360]
        height = sp.parse_short()
        sats = sp.parse_byte()
        hdop = sp.parse_short() / 100
        return WialonCombineDataPacketPositionData(
            lat=lat,
            lon=lon,
            speed=speed,
            course=course,
            height=height,
            sats=sats,
            hdop=hdop,
        )

    def _parse_driver_message(self, sp):
        text = sp.parse_string()
        return WialonCombineDataPacketDriverMessage(
            text=text,
        )


class WialonCombineDataPacketCustomParameters(object):

    __slots__ = ['params']

    def __init__(self, params):
        self.params = params

    def to_dict(self):
        d = {
            'type': 'custom_parameters',
            'params': self.params,
        }
        return d


class WialonCombineDataPacketPositionData(object):

    __slots__ = ['lat', 'lon', 'speed', 'course', 'height', 'sats', 'hdop']

    def __init__(self, lat, lon, speed, course, height, sats, hdop):
        self.lat = lat
        self.lon = lon
        self.speed = speed
        self.course = course
        self.height = height
        self.sats = sats
        self.hdop = hdop

    def to_dict(self):
        d = {
            'type': 'position_data',
        }
        for field in self.__slots__:
            d[field] = getattr(self, field)
        return d


class WialonCombineDataPacketDriverMessage(object):

    __slots__ = ['text']

    def __init__(self, text):
        self.text = text

    def to_dict(self):
        d = {
            'text': self.text,
        }
        return d
