import logging

from ..util.parsing import StreamParser
from .base import WialonCombinePacket


LOGGER = logging.getLogger(__name__)


class WialonCombineLoginPacket(WialonCombinePacket):

    __slots__ = ['id', 'pwd']

    def __init__(self, id_, pwd):
        self.id = id_
        self.pwd = pwd

    def __repr__(self):
        return '<WialonCombineLoginPacket: id={}; pwd={}>'.format(self.id, self.pwd)

    @classmethod
    def from_generic_packet(cls, generic_packet):
        parser = WialonCombineLoginPacketParser()
        return parser.parse(generic_packet.data)

    @classmethod
    def get_type(cls):
        return 'login'

    def to_dict(self):
        return {
            'id': self.id,
            'pwd': self.pwd,
        }


class WialonCombineLoginPacketParser(object):

    def parse(self, raw_data):
        sp = StreamParser(iter(raw_data))

        protocol_version = sp.parse_byte()
        if protocol_version != 1:
            raise RuntimeError('Unexpected protocol version: %s', protocol_version)

        id_, pwd = self._parse_credentials(sp)

        return WialonCombineLoginPacket(
            id_=id_,
            pwd=pwd,
        )

    def _parse_credentials(self, sp):
        flags = sp.parse_byte()

        id_type_mask = 0b11110000
        pwd_type_mask = 0b00001111

        id_type = (flags & id_type_mask) >> 4
        pwd_type = flags & pwd_type_mask

        id_parse_func = {
            1: sp.parse_short,
            2: sp.parse_int,
            3: sp.parse_long,
            4: sp.parse_string,
        }[id_type]
        id_ = id_parse_func()

        if pwd_type == 0:
            pwd = None
        else:
            pwd_parse_func = {
                1: sp.parse_short,
                2: sp.parse_int,
                3: sp.parse_long,
                4: sp.parse_string,
            }[pwd_type]
            pwd = pwd_parse_func()

        return id_, pwd
