import contextlib
import struct


class StreamParser(object):

    def __init__(self, stream):
        self.accumulated_data = bytearray()
        self._stream = stream

    def parse(self, nbytes, fmt):
        bs = bytearray()
        for _ in range(nbytes):
            bs.append(self._fetch_byte())
        value = self._unpack(fmt, bs)
        return value

    def parse_byte(self, accumulate=True):
        return self._fetch_byte(accumulate=accumulate)

    def parse_extensible_byte(self):
        bs = bytearray()
        bs.append(self._fetch_byte())
        if bs[0] & 1 << 7:
            bs[0] ^= 1 << 7
            value_type = 'H'
            bs.append(self._fetch_byte())
        else:
            value_type = 'B'
        value = self._unpack(value_type, bs)
        return value

    def parse_short(self):
        return self.parse(2, 'H')

    def parse_extensible_short(self):
        bs = bytearray()
        bs.append(self._fetch_byte())
        bs.append(self._fetch_byte())
        if bs[0] & 1 << 7:
            bs[0] ^= 1 << 7
            value_type = 'I'
            bs.append(self._fetch_byte())
            bs.append(self._fetch_byte())
        else:
            value_type = 'H'
        value = self._unpack(value_type, bs)
        return value

    def parse_int(self):
        return self.parse(4, 'I')

    def parse_long(self):
        return self.parse(8, 'Q')

    def parse_string(self):
        characters = []
        while True:
            byte = self.parse_byte()
            if byte == 0:
                break
            characters.append(byte)
        value = ''.join([chr(c) for c in characters])
        return value

    def _fetch_byte(self, accumulate=True):
        byte = next(self._stream)
        if accumulate:
            self.accumulated_data.append(byte)
        return byte

    def _unpack(self, type_, bs):
        fmt = '>{}'.format(type_)
        return struct.unpack(fmt, bs)[0]


class BufferReader(object):

    def __init__(self, buf):
        self.buf = buf

    def __len__(self):
        return sum([len(data) for data in self.buf])

    @contextlib.contextmanager
    def iterator(self):
        iterator = BufferReaderIterator(self.buf)
        try:
            yield iterator
        except Exception:
            raise
        finally:
            data = iterator.get_leftover_data()
            if data:
                self.buf.appendleft(data)


class BufferReaderIterator(object):

    def __init__(self, buf):
        self.buf = buf    # Sequence of bytestrings.
        self.data = None  # Currently in-process bytestring.
        self.pos = None   # Current position in self.data.

    def __iter__(self):
        return self

    def __next__(self):
        if not self.data or self.pos == len(self.data):
            self.data = None
            while self.buf:
                self.data = self.buf.popleft()
                if self.data:
                    break
            if not self.data:
                raise StopIteration
            self.pos = 0

        byte = self.data[self.pos]
        self.pos += 1

        return byte

    def get_leftover_data(self):
        return self.data[self.pos:] if self.data else b''
