import enum
import logging
import threading
import time
import uuid

from twisted.internet import defer, reactor


LOGGER = logging.getLogger(__name__)


class BaseCommand(object):

    def __init__(self, id_, imei, text, issued_at=None, timeout=25):
        self._id = id_
        self._imei = imei
        self._text = text
        self._replies = []
        self._result = None
        self._ready_ev = threading.Event()

        self._deferred = defer.Deferred()
        self._deferred.addTimeout(timeout, reactor)
        self._deferred.addErrback(self._on_timeout)

        if issued_at is None:
            issued_at = time.time()
        self._issued_at = issued_at

    @property
    def id(self):
        return self._id

    @property
    def imei(self):
        return self._imei

    @property
    def deferred(self):
        return self._deferred

    @property
    def text(self):
        return self._text

    @property
    def replies(self):
        return self._replies

    @property
    def issued_at(self):
        return self._issued_at

    def is_ready(self):
        return self._ready_ev.is_set()

    def handle_reply(self, text):
        reply = CommandReply(
            received_at=time.time(),
            text=text,
        )
        self._replies.append(reply)

        if text == 'command lost (busy)':
            result = CommandResult(
                status=CommandResult.Status.ERROR,
                code='command_lost',
            )
            self._set_result(result)
        else:
            self._handle_specific_reply(text)

    def _handle_specific_reply(self, text):
        raise NotImplementedError

    def to_dict(self):
        return {
            'id': self._id,
            'issued_at': self._issued_at,
            'text': self._text,
            'replies': [reply.to_dict() for reply in self._replies],
            'result': self._result.to_dict() if self.is_ready() else None,
        }

    def _on_timeout(self, failure):
        failure.trap(defer.TimeoutError)
        result = CommandResult(status=CommandResult.Status.TIMEOUT)
        self._set_result(result)
        return result

    def _set_result(self, result):
        if self._result is not None:
            LOGGER.warning(
                'command result already set: %s vs %s', self.to_dict(), result.to_dict()
            )
            return

        self._result = result
        self._ready_ev.set()
        try:
            self._deferred.callback(self._result)
        except Exception:
            LOGGER.exception('failed to invoke command callback')

    def _generate_id(self):
        # Telematics software strips leading zero and convert command id to uppercase.
        id_ = uuid.uuid4().hex[:8].upper()
        id_ = id_.replace('0', '1')
        return id_


class CommandReply(object):

    def __init__(self, received_at, text):
        self.received_at = received_at
        self.text = text

    def to_dict(self):
        return {
            'received_at': self.received_at,
            'text': self.text,
        }


class CommandResult(object):

    class Status(enum.Enum):
        OK = 'ok'
        ERROR = 'error'
        TIMEOUT = 'timeout'

    def __init__(self, status, code=None):
        self.status = status
        self.code = code

    def to_dict(self):
        return {
            'status': self.status.value,
            'code': self.code,
        }
