import logging

from .base import BaseCommand, CommandResult


LOGGER = logging.getLogger(__name__)


class WarmingCommand(BaseCommand):

    _ERROR_REPLIES = {
        'break, driver door is open': 'driver_door_open',
        'break, pass door is open': 'pass_door_open',
        'break, left rear door is open': 'left_rear_door_open',
        'break, right rear door is open': 'right_rear_door_open',
        'break, hood door is open': 'hood_door_open',
        'break, trunk door is open': 'trunk_door_open',
        'break, wrong key position': 'ignition_on',
        'break, wrong selector position': 'selector_not_parking',
        'break, hood or trunk is open': 'hood_or_trunk_open',
        'break, ignition did not appeared...': 'ignition_failed',
        'break, transmission is not at parking': 'selector_not_parking',
        'engine run fail': 'engine_failed',
    }

    def __init__(self, *args, id_=None, **kwargs):
        if id_ is None:
            id_ = self._generate_id()
        super().__init__(*args, id_=id_, text='yadrive_warming', **kwargs)

    def _handle_specific_reply(self, text):
        status = None
        code = None
        if text in ('engine is on', 'break, engine is on'):
            status = CommandResult.Status.OK
        elif text in self._ERROR_REPLIES:
            status = CommandResult.Status.ERROR
            code = self._ERROR_REPLIES[text]
        elif text.startswith('break'):
            LOGGER.warning('unknown warming break: %s', repr(text))
            status = CommandResult.Status.ERROR
            code = 'unknown'

        if status is not None:
            result = CommandResult(
                status=status,
                code=code,
            )
            self._set_result(result)


class StopWarmingCommand(BaseCommand):

    _FINAL_REPLIES = {
        'engine stop done',
        'stop warning already in progress',
        'no active warming',
        'engine stop event by stop warming cmd',
        'engine stop done'
    }

    def __init__(self, *args, id_=None, **kwargs):
        if id_ is None:
            id_ = self._generate_id()
        super().__init__(*args, id_=id_, text='yadrive_stop_warming', **kwargs)

    def _handle_specific_reply(self, text):
        if text in self._FINAL_REPLIES:
            result = CommandResult(
                status=CommandResult.Status.OK,
                code=None,
            )
            self._set_result(result)
