import json
import logging

from twisted.internet.defer import CancelledError
from twisted.web.resource import NoResource, Resource
from twisted.web.server import NOT_DONE_YET
from twisted.web.server import Site

import cars.telematics.backend.protocol_registry
from ..commands.doors import CloseDoorsCommand, OpenDoorsCommand
from ..commands.end_of_lease import EndOfLeaseCommand
from ..commands.hood import LockHoodCommand, UnlockHoodCommand
from ..commands.simple import SimpleCommand
from ..commands.start_of_lease import StartOfLeaseCommand
from ..commands.warming import StopWarmingCommand, WarmingCommand


LOGGER = logging.getLogger(__name__)


class CarsAPIResource(Resource):

    def __init__(self, *args, server, **kwargs):
        super().__init__(*args, **kwargs)
        self._server = server

    def getChild(self, name, request):
        if name == b'':
            return NoResource()
        try:
            car_id = int(name.decode('utf-8'))
        except ValueError:
            return NoResource()
        return CarAPIResource(server=self._server, car_id=car_id)


class CarAPIResource(Resource):

    def __init__(self, *args, server, car_id, **kwargs):
        super().__init__(*args, **kwargs)
        self._server = server
        self._car_id = car_id

    def getChild(self, name, request):
        if name == b'command':
            return CommandCarAPIResource(server=self._server, car_id=self._car_id)
        return NoResource()


class CommandCarAPIResource(Resource):

    def __init__(self, *args, server, car_id, **kwargs):
        super().__init__(*args, **kwargs)
        self._server = server
        self._car_id = car_id

    def getChild(self, name, request):
        resource_class = {
            b'custom': CustomCommandCarAPIResource,
            b'getserverconfig': GetServerConfigCommandCarAPIResource,
            b'blink': BlinkCommandCarAPIResource,
            b'closealldoors': CloseAllDoorsCommandCarAPIResource,
            b'openalldoors': OpenAllDoorsCommandCarAPIResource,
            b'startengine': StartEngineCommandCarAPIResource,
            b'stopengine': StopEngineCommandCarAPIResource,
            b'startlease': StartOfLeaseCommandCarAPIResource,
            b'endlease': EndOfLeaseCommandCarAPIResource,
            b'warming': WarmingCommandCarAPIResource,
            b'stopwarming': StopWarmingCommandCarAPIResource,
            b'lockhood': LockHoodCommandCarAPIResource,
            b'unlockhood': UnlockHoodCommandCarAPIResource,
        }.get(name)

        if resource_class is None:
            return NoResource()

        resource = resource_class(server=self._server, car_id=self._car_id)

        return resource


class BaseCommandCarAPIResource(Resource):

    isLeaf = True

    def __init__(self, *args, server, car_id, **kwargs):
        super().__init__(*args, **kwargs)
        self._server = server
        self._car_id = car_id

    def _get_command(self, request):
        raise NotImplementedError

    def render_POST(self, request):
        try:
            command = self._get_command(request)
        except Exception:
            request.setResponseCode(400)
            return b''

        self._handle_command(request, command)

        return NOT_DONE_YET

    def _handle_command(self, request, command):

        def on_not_found(failure):
            failure.trap(cars.telematics.backend.protocol_registry.ProtocolRegistry.NotFound)
            request.setResponseCode(404)
            request.finish()
            return None

        def on_cancel(failure):
            failure.trap(CancelledError)
            LOGGER.warning('command canceled: %s to %s', repr(command), self._car_id)
            return None

        def on_error(failure):
            failure.trap(Exception)
            exc_info = (failure.type, failure.value, failure.getTracebackObject())
            LOGGER.error(
                'failed to handle command %s to %s',
                repr(command),
                self._car_id,
                exc_info=exc_info,
            )
            request.setResponseCode(500)
            request.finish()
            return None

        def on_response_failed(failure, deferred_command):
            LOGGER.info('response failed: %s', failure.getErrorMessage())
            deferred_command.cancel()
            return None

        def on_result(result):
            # Return if an error occured and an errback returned None.
            if result is None:
                return

            data = {
                'status': result.status.value,
                'code': result.code,
            }
            data_json = json.dumps(data, ensure_ascii=False)

            request.setResponseCode(200)
            request.write(data_json.encode('utf-8'))
            request.finish()

        d = self._server.handle_command(command=command)
        (
            d
            .addErrback(on_not_found)
            .addErrback(on_cancel)
            .addErrback(on_error)
            .addCallback(on_result)
        )

        request.notifyFinish().addErrback(on_response_failed, d)

        return d


class BaseSimpleCommandCarAPIResource(BaseCommandCarAPIResource):

    def _get_command(self, request):
        command_text = self._get_command_text(request)
        return SimpleCommand(
            imei=self._car_id,
            text=command_text,
            timeout=self._server.timeout,
        )

    def _get_command_text(self, request):
        raise NotImplementedError


class CustomCommandCarAPIResource(BaseSimpleCommandCarAPIResource):
    def _get_command_text(self, request):
        data_str = request.content.read().decode('utf-8')
        data = json.loads(data_str)
        command = data['command']
        command = command.replace('\\n', '\n').replace('\\r', '\r')
        return command

class GetServerConfigCommandCarAPIResource(BaseSimpleCommandCarAPIResource):
    def _get_command_text(self, _):
        return 'server?'

class BlinkCommandCarAPIResource(BaseSimpleCommandCarAPIResource):
    def _get_command_text(self, _):
        return 'can_blinkerflasing'

class StartEngineCommandCarAPIResource(BaseSimpleCommandCarAPIResource):
    def _get_command_text(self, _):
        return 'can_startengine'

class StopEngineCommandCarAPIResource(BaseSimpleCommandCarAPIResource):
    def _get_command_text(self, _):
        return 'can_stopengine'

class OpenAllDoorsCommandCarAPIResource(BaseCommandCarAPIResource):
    def _get_command(self, _):
        return OpenDoorsCommand(imei=self._car_id, timeout=self._server.timeout)

class CloseAllDoorsCommandCarAPIResource(BaseCommandCarAPIResource):
    def _get_command(self, _):
        return CloseDoorsCommand(imei=self._car_id, timeout=self._server.timeout)

class StartOfLeaseCommandCarAPIResource(BaseCommandCarAPIResource):
    def _get_command(self, _):
        return StartOfLeaseCommand(imei=self._car_id, timeout=self._server.timeout)

class EndOfLeaseCommandCarAPIResource(BaseCommandCarAPIResource):
    def _get_command(self, _):
        return EndOfLeaseCommand(imei=self._car_id, timeout=self._server.timeout)

class WarmingCommandCarAPIResource(BaseCommandCarAPIResource):
    def _get_command(self, _):
        return WarmingCommand(imei=self._car_id, timeout=self._server.timeout)

class StopWarmingCommandCarAPIResource(BaseCommandCarAPIResource):
    def _get_command(self, _):
        return StopWarmingCommand(imei=self._car_id, timeout=self._server.timeout)

class LockHoodCommandCarAPIResource(BaseCommandCarAPIResource):
    def _get_command(self, _):
        return LockHoodCommand(imei=self._car_id, timeout=self._server.timeout)

class UnlockHoodCommandCarAPIResource(BaseCommandCarAPIResource):
    def _get_command(self, _):
        return UnlockHoodCommand(imei=self._car_id, timeout=self._server.timeout)


class TelematicsFrontendFactory(Site):

    def __init__(self, *args, server, **kwargs):
        root = self.make_root_resource(server=server)
        super().__init__(*args, resource=root, **kwargs)
        self._server = server

    def make_root_resource(self, server):
        root = Resource()
        root_api = Resource()
        root_api_cars = CarsAPIResource(server=server)

        root.putChild(b'api', root_api)
        root_api.putChild(b'cars', root_api_cars)

        return root
