import logging
import threading

from twisted.internet import reactor
from twisted.python import log

import cars.settings
from .backend.packet_handlers import (
    LoggerPacketHandler,
    SaasDriveCarSubmitterPacketHandler,
    SaasDriveNavigatorPrestableRidingSubmitterPacketHandler,
    SaasDriveNavigatorStableRidingSubmitterPacketHandler,
    SaasDriveRidingSubmitterPacketHandler,
    SignalCollectorPacketHandler,
)
from .backend.runner import TCP6TelematicsBackendServerRunner
from .backend.server import WialonCombineTelematicsBackendServer
from .frontend.server import HTTPTelematicsFrontendServer
from .push_client import TelematicsPushClientLogger


LOGGER = logging.getLogger(__name__)


class TelematicsServer(object):

    def __init__(self, backend_port, frontend_port, frontend_timeout):
        self._push_client = TelematicsPushClientLogger(
            filename=cars.settings.TELEMATICS['push_client']['filename'],
            lock=threading.Lock(),
        )

        logger_packet_handler = LoggerPacketHandler(push_client=self._push_client)
        saas_drive_car_packer_handler = SaasDriveCarSubmitterPacketHandler.from_settings()
        saas_drive_riding_packet_handler = SaasDriveRidingSubmitterPacketHandler.from_settings()
        saas_drive_navi_riding_packer_handler_prestable = (
            SaasDriveNavigatorPrestableRidingSubmitterPacketHandler.from_settings()
        )
        saas_drive_navi_riding_packer_handler_stable = (
            SaasDriveNavigatorStableRidingSubmitterPacketHandler.from_settings()
        )
        signal_collector_packet_handler = SignalCollectorPacketHandler.from_settings()
        packet_handlers = [
            logger_packet_handler,
            signal_collector_packet_handler,
        ]

        self._backend = WialonCombineTelematicsBackendServer(
            packet_handlers=packet_handlers,
            push_client=self._push_client,
        )
        self._backend_runner = TCP6TelematicsBackendServerRunner(
            reactor=reactor,
            port=backend_port,
            factory=self._backend.get_factory(),
        )

        self._frontend = HTTPTelematicsFrontendServer(
            base_server=self,
            reactor=reactor,
            port=frontend_port,
            timeout=frontend_timeout,
        )

    def run(self, run_reactor=True):
        self._backend_runner.run()
        self._frontend.run()
        if run_reactor:
            reactor.run()  # pylint: disable=no-member

    def stop(self):
        self._frontend.stop()
        self._backend_runner.stop()

    def send_command(self, command):
        LOGGER.info('sending %s to car %s', command.to_dict(), repr(command.imei))
        self._backend.send_command(command)


def main(backend_port, frontend_port):
    observer = log.PythonLoggingObserver()
    observer.start()

    server = TelematicsServer(
        backend_port=backend_port,
        frontend_port=frontend_port,
        frontend_timeout=cars.settings.TELEMATICS['frontend']['timeout'],
    )
    server.run()
