# -*- coding: utf-8 -*-

from datetime import datetime
from time import sleep
from random import randint, choice
from threading import Thread
from logging import getLogger
from cars import settings
from cars.core.util import make_yt_client
from .daemon import Daemon
from ..helpers import AsyncYtWriter, AsyncSaasCarsWriter
from ..providers import Anytime, Account


LOGGER = getLogger(__name__)


class AnytimeDaemon(Daemon):
    code = 'anytime'

    CARS_TABLE = 'data/thief/anytime/1d/%Y-%m-%d'

    ACCOUNTS_TABLE = 'data/thief/anytime/accounts'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._writer = AsyncYtWriter(make_yt_client('data'))
        self._saas_writer = AsyncSaasCarsWriter()
        self._provider = Anytime()
        self._accounts = {}
        self._accounts_delay = {}

    def _reload_accounts(self, region):
        self._accounts[region] = [
            Account(
                region=region,
                data=None,
                created=None,
                updated=None,
            )
        ]
        self._accounts_delay[region] = 10

    def _get_random_account(self, region):
        if self._accounts_delay.get(region, 0) <= 0:
            self._reload_accounts(region)
        self._accounts_delay[region] -= 1
        return choice(self._accounts[region])

    def _do_tick(self):
        threads = [
            Thread(target=self._region_loop, args=('msk',)),
        ]
        self._writer.open()
        self._saas_writer.open()
        for thread in threads:
            thread.start()
        for thread in threads:
            thread.join()
        self._writer.close()
        self._saas_writer.close()

    def _solomon_set_value(self, key, value):
        self.solomon.set_value(
            '{}.{}'.format(self.get_solomon_sensor_prefix(), key),
            value
        )

    def _region_loop(self, region):
        while True:
            try:
                account = self._get_random_account(region)
                try:
                    car_states = self._provider.load_car_states(account)
                    self._solomon_set_value('cars_load_failure', 0)
                except Exception as exception:
                    LOGGER.exception(exception)
                    self._solomon_set_value('cars_load_failure', 1)
                    sleep(randint(40, 80))
                    continue
                time = datetime.utcnow()
                timestamp = int(time.timestamp())
                path = '{}/{}'.format(
                    settings.EXPORT['home_directory'],
                    time.strftime(self.CARS_TABLE)
                )
                self._writer.set_schema(path, self._provider._get_schema())
                self._saas_writer.write(self.code, region, timestamp, car_states)
                for car_state in car_states:
                    self._writer.write(
                        path,
                        self._provider._value_to_row(car_state)
                    )
                self._solomon_set_value(
                    'total_cars.{}'.format(region),
                    len(car_states)
                )
                sleep(randint(40, 80))
            except BaseException:
                sleep(randint(40, 80))
