# -*- coding: utf-8 -*-

from datetime import datetime
from time import sleep
from random import randint, choice
from threading import Thread
from logging import getLogger
from cars import settings
from cars.core.util import make_yt_client
from .daemon import Daemon
from ..helpers import AsyncYtWriter, AsyncSaasCarsWriter
from ..providers import Belkacar, Account


LOGGER = getLogger(__name__)


class BelkacarDaemon(Daemon):
    code = 'belkacar'

    CARS_TABLE = 'data/thief/belkacar/1d/%Y-%m-%d'

    ACCOUNTS_TABLE = 'data/thief/belkacar/accounts'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._writer = AsyncYtWriter(make_yt_client('data'))
        self._saas_writer = AsyncSaasCarsWriter()
        self._provider = Belkacar()
        self._accounts = {}
        self._accounts_delay = {}

    def _reload_accounts(self, region):
        self._accounts[region] = []
        path = '{}/{}'.format(
            settings.EXPORT['home_directory'],
            self.ACCOUNTS_TABLE
        )
        try:
            for row in self._yt.read_table(path):
                if row['region'] == region:
                    self._accounts[region].append(
                        Account(
                            region=row['region'],
                            data=row['data'],
                            created=row['created'],
                            updated=row['updated'],
                        )
                    )
            self._accounts_delay[region] = 10
        except Exception as exception:
            LOGGER.exception(exception)

    def _get_random_account(self, region):
        if self._accounts_delay.get(region, 0) <= 0:
            self._reload_accounts(region)
        self._accounts_delay[region] -= 1
        return choice(self._accounts[region])

    def _do_tick(self):
        threads = [
            Thread(target=self._region_loop, args=('msk',)),
            Thread(target=self._update_accounts_loop),
        ]
        self._writer.open()
        self._saas_writer.open()
        for thread in threads:
            thread.start()
        for thread in threads:
            thread.join()
        self._writer.close()
        self._saas_writer.close()

    def _solomon_set_value(self, key, value):
        self.solomon.set_value(
            '{}.{}'.format(self.get_solomon_sensor_prefix(), key),
            value
        )

    def _update_accounts_loop(self):
        while True:
            path = '{}/{}'.format(
                settings.EXPORT['home_directory'],
                self.ACCOUNTS_TABLE
            )
            curr_rows = list(self._yt.read_table(path))
            new_rows = []
            for row in curr_rows:
                account = Account(
                    region=row['region'],
                    data=row['data'],
                    created=row['created'],
                    updated=row['updated'],
                )
                try:
                    if self._provider.update_account(account):
                        self._solomon_set_value('account_update_success', 1)
                        sleep(randint(120, 180))
                except Exception as exception:
                    LOGGER.exception(exception)
                    self._solomon_set_value('account_update_failure', 1)
                new_rows.append({
                    'region': account.region,
                    'data': account.data,
                    'created': account.created,
                    'updated': account.updated,
                })
            for i in range(1, 60):
                try:
                    self._yt.write_table(path, new_rows)
                    sleep(120)
                    break
                except Exception:
                    pass
            sleep(3600)

    def _region_loop(self, region):
        while True:
            try:
                account = self._get_random_account(region)
                try:
                    car_states = self._provider.load_car_states(account)
                    self._solomon_set_value('cars_load_failure', 0)
                except Exception as exception:
                    LOGGER.exception(exception)
                    self._solomon_set_value('cars_load_failure', 1)
                    sleep(randint(40, 80))
                    continue
                time = datetime.utcnow()
                timestamp = int(time.timestamp())
                path = '{}/{}'.format(
                    settings.EXPORT['home_directory'],
                    time.strftime(self.CARS_TABLE)
                )
                self._writer.set_schema(path, self._provider._get_schema())
                self._saas_writer.write(self.code, region, timestamp, car_states)
                for car_state in car_states:
                    self._writer.write(
                        path,
                        self._provider._value_to_row(car_state)
                    )
                self._solomon_set_value(
                    'total_cars.{}'.format(region),
                    len(car_states)
                )
                sleep(randint(40, 80))
            except BaseException:
                sleep(randint(40, 80))
