# -*- coding: utf-8 -*-

from datetime import datetime, timedelta
from time import sleep
from random import randint, choice
from threading import Thread
from collections import defaultdict
from logging import getLogger
from cars import settings
from cars.core.util import make_yt_client
from copy import deepcopy
from .daemon import Daemon
from ..helpers import AsyncYtWriter, AsyncSaasCarsWriter
from ..providers import Delimobil, CarState


LOGGER = getLogger(__name__)


class DelimobilDaemon(Daemon):
    code = 'delimobil'

    CARS_TABLE = 'data/thief/delimobil/1d/%Y-%m-%d'
    CARS_TABLE_SCHEMA = [
        {'name': 'car_id',    'type': 'string', 'required': True},
        {'name': 'position',  'type': 'any'                     },
        {'name': 'region',    'type': 'string'                  },
        {'name': 'operator',  'type': 'string'                  },
        {'name': 'tariffs',   'type': 'any'                     },
        {'name': 'fuel',      'type': 'double'                  },
        {'name': 'data',      'type': 'any'                     },
        {'name': 'timestamp', 'type': 'int64'                   },
    ]

    USERS_TABLE = 'data/thief/delimobil/users'

    REGION_TIMEDELTA = {  # Timedelta in minutes
        'msk': 3 * 60,
        'spb': 3 * 60,
        'ufa': 3 * 60,
        'ekb': 3 * 60,
    }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._writer = AsyncYtWriter(make_yt_client('data'))
        self._saas_writer = AsyncSaasCarsWriter()
        self._users = {}
        self._users_delay = {}
        self._msk_user = Delimobil()  # kostyl for Moscow

    def _do_tick(self):
        jobs = [
            Thread(target=self._region_loop, args=('msk',)),
            Thread(target=self._region_loop, args=('spb',)),
            Thread(target=self._region_loop, args=('ufa',)),
            Thread(target=self._region_loop, args=('ekb',)),
        ]
        self._writer.open()
        self._saas_writer.open()
        for job in jobs:
            job.start()
        for job in jobs:
            job.join()
        self._writer.close()
        self._saas_writer.close()

    def _reload_users(self, region):
        self._users[region] = []
        path = '{}/{}'.format(
            settings.EXPORT['home_directory'],
            self.USERS_TABLE
        )
        try:
            for row in self._yt.read_table(path):
                if row['region'] == region:
                    self._users[region].append(Delimobil(row['token']))
            self._users_delay[region] = 10
        except Exception as exception:
            LOGGER.exception(exception)

    def _get_provider(self, region):
        if self._users_delay.get(region, 0) <= 0:
            self._reload_users(region)
        self._users_delay[region] -= 1
        return choice(self._users[region])

    def _solomon_set_value(self, key, value):
        self.solomon.set_value(
            '{}.{}'.format(self.get_solomon_sensor_prefix(), key),
            value
        )

    def _region_loop(self, region):
        delay = 0
        tariffs = None
        while True:
            try:
                try:
                    provider = self._get_provider(region)
                except IndexError:
                    sleep(45)
                    continue
                # Summary duration is about ~ 1 min ([30, 90] sec)
                if delay <= 0:
                    # We do not need to always update tariffs
                    # So, update them every ~ 10 min
                    try:
                        tariffs = provider.load_tariffs()
                        self._solomon_set_value('tariffs_load_failure', 0)
                        delay = 10
                    except Exception as exception:
                        LOGGER.exception(exception)
                        self._solomon_set_value('tariffs_load_failure', 1)
                delay -= 1
                sleep(randint(15, 45))
                if tariffs is None:
                    self._solomon_set_value('cars_load_failure', 1)
                    continue
                try:
                    cars = provider.load_cars()
                    self._solomon_set_value('cars_load_failure', 0)
                except Exception as exception:
                    LOGGER.exception(exception)
                    self._solomon_set_value('cars_load_failure', 1)
                    sleep(randint(15, 45))
                    continue
                time = datetime.utcnow()
                timestamp = int(time.timestamp())
                local_time = time + timedelta(
                    minutes=self.REGION_TIMEDELTA[region]
                )
                default_tariffs = [
                    self._parse_tariff(tariffs['tariff'], local_time)
                ]
                car_tariffs = self._parse_tariffs(tariffs, local_time)
                path = '{}/{}'.format(
                    settings.EXPORT['home_directory'],
                    time.strftime(self.CARS_TABLE)
                )
                self._writer.set_schema(path, self.CARS_TABLE_SCHEMA)
                car_tariff_not_found = 0
                car_states = list()
                for car in cars['cars']:
                    name = car['model']['name']
                    if name not in car_tariffs:
                        LOGGER.warning(
                            'Tariff for car \'{}\' is not found'.format(name)
                        )
                        car_tariff_not_found += 1
                        row_tariffs = default_tariffs
                    else:
                        row_tariffs = car_tariffs[name]
                    self._writer.write(
                        path,
                        {
                            'timestamp': timestamp,
                            'position': (car['lat'], car['lon']),
                            'region': region,
                            'operator': 'delimobil',
                            'tariffs': row_tariffs,
                            'car_id': str(car['id']),
                            'fuel': round(int(car['fuel'].rstrip(' %')) / 100, 3),
                            'data': car['model'],
                        }
                    )
                    data = deepcopy(car['model'])
                    if 'name_full' in data:
                        data['model'] = data['name_full']
                    data['fuel_level'] = int(car['fuel'].rstrip(' %'))
                    parking_price = None
                    reservation_price = None
                    riding_price = None
                    for tariff in row_tariffs:
                        if 'park_price' in tariff:
                            parking_price = tariff['park_price']
                        if 'reserve_price' in tariff:
                            reservation_price = tariff['reserve_price']
                        if 'ride_price' in tariff:
                            riding_price = tariff['ride_price']
                    if parking_price:
                        data['parking_price'] = parking_price
                    if reservation_price:
                        data['reservation_price'] = reservation_price
                    if riding_price:
                        data['riding_price'] = riding_price
                    car_state = CarState(
                        car_id=str(car['id']),
                        timestamp=timestamp,
                        region=region,
                        operator=self.code,
                        position=(car['lat'], car['lon']),
                        tariffs=row_tariffs,
                        data=data,
                    )
                    car_states.append(car_state)
                self._saas_writer.write(self.code, region, timestamp, car_states)
                self._solomon_set_value(
                    'car_tariff_not_found',
                    car_tariff_not_found
                )
                self._solomon_set_value(
                    'total_cars.{}'.format(region),
                    len(cars['cars'])
                )
                sleep(60)
            except BaseException as exc:
                LOGGER.exception(exc)
                sleep(120)

    @staticmethod
    def _get_model_name(name):
        if name == 'Volkswagen Polo':
            return 'VW Polo'
        return name

    @classmethod
    def _parse_tariffs(cls, tariffs, time):
        car_tariffs = {}
        for model in tariffs['models']:
            car_tariffs[cls._get_model_name(model['name'])] = [
                cls._parse_tariff(model['tariff'], time)
            ]
        return car_tariffs

    @classmethod
    def _parse_tariff(cls, tariff, time):
        ride_price = tariff['trip_extend']['value']
        park_price = tariff['park_extend']['value']
        daytime = (time.hour * 60 + time.minute) * 60
        for interval in tariff['trip_intervals']:
            start = cls._parse_daytime(interval['from'])
            finish = cls._parse_daytime(interval['to'])
            if start < finish and start <= daytime and daytime <= finish or \
                start > finish and (start <= daytime or daytime <= finish):
                ride_price = interval['price_extend']['value']
        for interval in tariff['park_intervals']:
            start = cls._parse_daytime(interval['from'])
            finish = cls._parse_daytime(interval['to'])
            if start < finish and start <= daytime and daytime <= finish or \
                start > finish and (start <= daytime or daytime <= finish):
                park_price = interval['price_extend']['value']
        return {
            'name': tariff['name'],
            'ride_price': ride_price,
            'park_price': park_price,
            'reserve_price': tariff['reserve_extend']['value'],
            'day': {
                'price': tariff['day_extend']['value'],
                'time': (24 * 60 - 1) * 60,
                'mileage': tariff['day_max_mileage_extend']['value'],
                'over_time_price': 0,
                'over_mileage_price':
                    tariff['day_over_mileage_extend']['value'],
            }
        }

    @staticmethod
    def _parse_daytime(value):
        hour, minute = map(int, value.split(':'))
        return (hour * 60 + minute) * 60
