# -*- coding: utf-8 -*-

from datetime import datetime
from time import sleep
from random import randint, choice
from threading import Thread
from logging import getLogger
from cars import settings
from cars.core.util import make_yt_client
from .daemon import Daemon
from ..helpers import AsyncYtWriter, make_proxy_session
from ..providers import Urbi, Account


LOGGER = getLogger(__name__)


class UrbiDaemon(Daemon):
    code = 'urbi'

    CARS_TABLE = 'data/thief/urbi/1d/%Y-%m-%d'

    ACCOUNTS_TABLE = 'data/thief/urbi/accounts'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._writer = AsyncYtWriter(make_yt_client('data'))
        self._provider = Urbi()
        self._accounts = {}
        self._accounts_delay = {}
        self._selected_account = {}
        self._selected_delay = {}

    def _reload_accounts(self, region):
        path = '{}/{}'.format(
            settings.EXPORT['home_directory'],
            self.ACCOUNTS_TABLE
        )
        try:
            new_accounts = []
            yt = make_yt_client('data')
            for row in yt.read_table(path):
                if row['region'] == region:
                    account = Account(
                        region=row['region'],
                        data=row['data'],
                        created=row['created'],
                        updated=row['updated'],
                    )
                    account._session = make_proxy_session(row['data']['proxy'])
                    new_accounts.append(account)
            self._accounts[region] = new_accounts
            self._accounts_delay[region] = 15
        except Exception as exception:
            LOGGER.exception(exception)

    def _get_random_account(self, region):
        if self._accounts_delay.get(region, 0) <= 0:
            self._reload_accounts(region)
        self._accounts_delay[region] -= 1
        if self._selected_delay.get(region, 0) <= 0:
            self._selected_account[region] = choice(self._accounts[region])
            self._selected_delay[region] = 10
        self._selected_delay[region] -= 1
        return self._selected_account[region]

    def _do_tick(self):
        threads = [
            Thread(target=self._region_loop, args=('amsterdam:car2go',)),
            Thread(target=self._region_loop, args=('austin:car2go',)),
            Thread(target=self._region_loop, args=('berlin:car2go',)),
            Thread(target=self._region_loop, args=('berlin:drivenow',)),
            Thread(target=self._region_loop, args=('berlin:driveby',)),
            # Thread(target=self._region_loop, args=('calgari:car2go',)),
            Thread(target=self._region_loop, args=('catania:enjoy',)),
            Thread(target=self._region_loop, args=('cologne:car2go',)),
            Thread(target=self._region_loop, args=('cologne:drivenow',)),
            Thread(target=self._region_loop, args=('columbus:car2go',)),
            Thread(target=self._region_loop, args=('denver:car2go',)),
            Thread(target=self._region_loop, args=('dusseldorf:car2go',)),
            Thread(target=self._region_loop, args=('dusseldorf:drivenow',)),
            Thread(target=self._region_loop, args=('firenze:car2go',)),
            Thread(target=self._region_loop, args=('firenze:enjoy',)),
            Thread(target=self._region_loop, args=('firenze:sharengo',)),
            Thread(target=self._region_loop, args=('frankfurt:car2go',)),
            Thread(target=self._region_loop, args=('hamburg:car2go',)),
            Thread(target=self._region_loop, args=('hamburg:drivenow',)),
            Thread(target=self._region_loop, args=('helsinki:gonow',)),
            Thread(target=self._region_loop, args=('kobenhavn:drivenow',)),
            # Thread(target=self._region_loop, args=('london:drivenow',)),
            Thread(target=self._region_loop, args=('madrid:car2go',)),
            Thread(target=self._region_loop, args=('madrid:emov',)),
            Thread(target=self._region_loop, args=('milano:car2go',)),
            Thread(target=self._region_loop, args=('milano:drivenow',)),
            Thread(target=self._region_loop, args=('milano:enjoy',)),
            Thread(target=self._region_loop, args=('milano:sharengo',)),
            Thread(target=self._region_loop, args=('modena:sharengo',)),
            Thread(target=self._region_loop, args=('montreal:car2go',)),
            Thread(target=self._region_loop, args=('munich:car2go',)),
            Thread(target=self._region_loop, args=('munich:drivenow',)),
            Thread(target=self._region_loop, args=('newyork:car2go',)),
            Thread(target=self._region_loop, args=('portland:car2go',)),
            Thread(target=self._region_loop, args=('roma:car2go',)),
            Thread(target=self._region_loop, args=('roma:enjoy',)),
            Thread(target=self._region_loop, args=('roma:sharengo',)),
            Thread(target=self._region_loop, args=('seattle:car2go',)),
            Thread(target=self._region_loop, args=('stuttgart:car2go',)),
            Thread(target=self._region_loop, args=('torino:car2go',)),
            Thread(target=self._region_loop, args=('torino:enjoy',)),
            Thread(target=self._region_loop, args=('toronto:car2go',)),
            Thread(target=self._region_loop, args=('vancouver:car2go',)),
            Thread(target=self._region_loop, args=('washingtondc:car2go',)),
            Thread(target=self._region_loop, args=('wien:car2go',)),
            Thread(target=self._region_loop, args=('wien:drivenow',)),
        ]
        self._writer.open()
        for thread in threads:
            thread.start()
        for thread in threads:
            thread.join()
        self._writer.close()

    def _solomon_set_value(self, key, value):
        self.solomon.set_value(
            '{}.{}'.format(self.get_solomon_sensor_prefix(), key),
            value
        )

    def _region_loop(self, region):
        sleep(randint(0, 60))
        while True:
            try:
                account = self._get_random_account(region)
                try:
                    car_states = self._provider.load_car_states(account)
                    self._solomon_set_value('cars_load_failure', 0)
                except Exception as exception:
                    LOGGER.exception(exception)
                    self._solomon_set_value('cars_load_failure', 1)
                    self._selected_delay[region] = 0
                    sleep(randint(40, 80) * 5)
                    continue
                time = datetime.utcnow()
                path = '{}/{}'.format(
                    settings.EXPORT['home_directory'],
                    time.strftime(self.CARS_TABLE)
                )
                self._writer.set_schema(path, self._provider._get_schema())
                for car_state in car_states:
                    self._writer.write(
                        path,
                        self._provider._value_to_row(car_state)
                    )
                self._solomon_set_value(
                    'total_cars.{}'.format(region),
                    len(car_states)
                )
                sleep(randint(40, 80) * 5)
            except BaseException:
                sleep(randint(40, 80) * 5)
