# -*- coding: utf-8 -*-

from datetime import datetime
from time import sleep
from random import randint, choice
from threading import Thread
from requests import Session
from logging import getLogger
from cars import settings
from collections import defaultdict
from cars.core.util import make_yt_client
from .daemon import Daemon
from ..helpers import AsyncYtWriter
from ..providers import YandexDrive, Account


LOGGER = getLogger(__name__)


class YandexDriveDaemon(Daemon):
    code = 'yandexdrive'

    CARS_TABLE = 'data/thief/yandexdrive/1d/%Y-%m-%d'

    ACCOUNTS_TABLE = 'data/thief/yandexdrive/accounts'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._writer = AsyncYtWriter(make_yt_client('data'))
        self._provider = YandexDrive()
        self._accounts = {}
        self._accounts_delay = {}

    def _reload_accounts(self, region):
        self._accounts[region] = []
        path = '{}/{}'.format(
            settings.EXPORT['home_directory'],
            self.ACCOUNTS_TABLE
        )
        try:
            for row in self._yt.read_table(path):
                if row['region'] == region:
                    account = Account(
                        region=row['region'],
                        data=row['data'],
                        created=row['created'],
                        updated=row['updated'],
                    )
                    account._session = Session()
                    self._accounts[region].append(account)
            self._accounts_delay[region] = 10
        except Exception as exception:
            LOGGER.exception(exception)

    def _get_random_account(self, region):
        if self._accounts_delay.get(region, 0) <= 0:
            self._reload_accounts(region)
        self._accounts_delay[region] -= 1
        return choice(self._accounts[region])

    def _do_tick(self):
        threads = [
            Thread(target=self._main_loop),
        ]
        self._writer.open()
        for thread in threads:
            thread.start()
        for thread in threads:
            thread.join()
        self._writer.close()

    def _solomon_set_value(self, key, value):
        self.solomon.set_value(
            '{}.{}'.format(self.get_solomon_sensor_prefix(), key),
            value
        )

    def _main_loop(self):
        while True:
            try:
                account = self._get_random_account('msk')
                try:
                    car_states = self._provider.load_car_states(account)
                    self._solomon_set_value('cars_load_failure', 0)
                except Exception as exception:
                    LOGGER.exception(exception)
                    self._solomon_set_value('cars_load_failure', 1)
                    sleep(randint(40, 80))
                    continue
                time = datetime.utcnow()
                path = '{}/{}'.format(
                    settings.EXPORT['home_directory'],
                    time.strftime(self.CARS_TABLE)
                )
                self._writer.set_schema(path, self._provider._get_schema())
                region_counts = defaultdict(int)
                for car_state in car_states:
                    self._writer.write(
                        path,
                        self._provider._value_to_row(car_state)
                    )
                    region_counts[car_state.region] += 1
                for key, value in region_counts.items():
                    self._solomon_set_value('total_cars.{}'.format(key), value)
                sleep(randint(40, 80))
            except BaseException:
                sleep(randint(40, 80))
