# -*- coding: utf-8 -*-

"""Helpers for thief

TODO: Move this code to core (DANGER)
"""

from queue import Queue, Empty
from threading import Thread
from time import sleep
from base64 import b64encode, b64decode
from collections import defaultdict
from random import uniform
from requests import Session
from requests.adapters import HTTPAdapter
from logging import getLogger
from retry import retry
from yt.wrapper import TablePath
from cars import settings
from cars.settings import THIEF_SAAS
from cars.proto.external_device_pb2 import (
    TExternalDevice, TExternalDeviceSnapshot,
)


LOGGER = getLogger(__name__)


class AnimalsHTTPAdapter(HTTPAdapter):
    def __init__(
        self, proxy_url, proxy_port, proxy_auth=(None, None),
        proxy_headers=None, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.proxy_url = proxy_url
        self.proxy_port = proxy_port
        if proxy_headers is not None:
            assert isinstance(proxy_headers, MutableMapping), 'proxy_headers must be a dict.'
        self._proxy_headers = proxy_headers if proxy_headers else {}
        self._proxy_auth_login = proxy_auth[0]
        self._proxy_auth_password = proxy_auth[1]

    @property
    def proxy_address(self):
        if self._proxy_auth_login and self._proxy_auth_password:
            return 'http://{}:{}@{}:{}'.format(
                self._proxy_auth_login,
                self._proxy_auth_password,
                self.proxy_url,
                self.proxy_port
            )
        return 'http://{}:{}'.format(self.proxy_url, self.proxy_port)

    def proxy_headers(self, proxy):
        headers = super(AnimalsHTTPAdapter, self).proxy_headers(proxy)
        if self._proxy_headers:
            headers.update(self._proxy_headers)
        return headers


class ProxyHTTPAdapter(HTTPAdapter):
    def __init__(
        self, proxy_url, proxy_headers=None, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.proxy_url = proxy_url
        if proxy_headers is not None:
            assert isinstance(proxy_headers, MutableMapping), 'proxy_headers must be a dict.'
        self._proxy_headers = proxy_headers if proxy_headers else {}

    @property
    def proxy_address(self):
        return self.proxy_url

    def proxy_headers(self, proxy):
        headers = super(ProxyHTTPAdapter, self).proxy_headers(proxy)
        if self._proxy_headers:
            headers.update(self._proxy_headers)
        return headers


def make_animals_session():
    """Create proxy session
    """
    adapter = AnimalsHTTPAdapter(
        proxy_url=settings.ANIMALS['host'],
        proxy_port=settings.ANIMALS['port'],
        proxy_auth=(
            settings.ANIMALS['auth']['login'],
            settings.ANIMALS['auth']['token'],
        )
    )
    session = Session()
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    session.proxies = {
        'http': adapter.proxy_address,
        'https': adapter.proxy_address,
    }
    return session


def make_proxy_session(url):
    """Create proxy session
    """
    adapter = ProxyHTTPAdapter(url)
    session = Session()
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    session.proxies = {
        'http': adapter.proxy_address,
        'https': adapter.proxy_address,
    }
    return session


class AsyncYtWriter(object):
    """Queued Yt table writer (append)

    Use this bad code, if you do not want to be
    locked on writing to several Yt tables
    """

    def __init__(self, client):
        self._client = client
        self._queue = Queue()
        self._running = False
        self._thread = None
        self._schemas = {}

    def open(self):
        self._running = True
        self._thread = Thread(target=self._thread_loop)
        self._thread.start()

    def close(self):
        self._running = False
        self._thread.join()

    def write(self, table, row):
        self._queue.put((table, row))

    def set_schema(self, table, schema):
        self._schemas[table] = schema

    def _thread_loop(self):
        while True:
            sleep(60)
            if not self._running:
                self._upload_rows()
                break
            else:
                self._upload_rows()

    def _upload_rows(self):
        table_rows = defaultdict(list)
        while True:
            try:
                table, row = self._queue.get_nowait()
                table_rows[table].append(row)
            except Empty:
                break
        for table, rows in table_rows.items():
            try:
                if not self._client.exists(table):
                    self._client.create(
                        'table',
                        path=table,
                        attributes={
                            'schema': self._schemas.get(table),
                        },
                        recursive=True,
                        ignore_existing=True,
                    )
                self._client.write_table(
                    TablePath(table, append=True),
                    iter(rows)
                )
            except Exception as exception:
                for row in rows:
                    self.write(table, row)
                LOGGER.exception(exception)


def get_region_point(name):
    if name == 'msk':
        name = 'moscow'  # kostyl for convenient format of Moscow region
    for zone in settings.REGISTRATION_GEOZONES:
        if zone['name'] == name:
            break
    if zone['name'] != name:
        raise RuntimeError(
            'Unable to generate point for specified region'
        )
    return (
        uniform(zone['min_lat'], zone['max_lat']),
        uniform(zone['min_lon'], zone['max_lon'])
    )


class AsyncSaasCarsWriter(object):
    def __init__(self):
        self._session = Session()
        self._queue = Queue()
        self._running = False
        self._thread = None

    def open(self):
        self._running = True
        self._thread = Thread(target=self._thread_loop)
        self._thread.start()

    def close(self):
        self._running = False
        self._thread.join()

    def write(self, operator, region, timestamp, car_states):
        self._queue.put((operator, region, timestamp, car_states))

    def _thread_loop(self):
        while self._running:
            try:
                operator, region, timestamp, cars = self._queue.get(timeout=5)
            except Empty:
                continue
            try:
                self._upload_cars(operator, region, timestamp, cars)
            except Exception as exc:
                LOGGER.exception(exc)

    def _upload_cars(self, operator, region, timestamp, car_states):
        snapshot = TExternalDeviceSnapshot()
        snapshot.Operator = operator
        snapshot.Region = region
        snapshot.Timestamp = timestamp
        for car_state in car_states:
            device = snapshot.Device.add()
            device.Id = car_state.car_id
            if 'number' in car_state.data:
                device.Number = car_state.data['number']
            if 'model' in car_state.data:
                device.Model = car_state.data['model']
            device.Latitude = car_state.position[0]
            device.Longitude = car_state.position[1]
            if 'fuel_level' in car_state.data:
                device.FuelLevel = car_state.data['fuel_level']
            if 'fuel_distance' in car_state.data:
                device.FuelDistance = car_state.data['fuel_distance']
            if 'parking_price' in car_state.data:
                device.ParkingPrice = int(car_state.data['parking_price'] * 100)
            if 'reservation_price' in car_state.data:
                device.ReservationPrice = int(car_state.data['reservation_price'] * 100)
            if 'riding_price' in car_state.data:
                device.RidingPrice = int(car_state.data['riding_price'] * 100)
        proto = b64encode(snapshot.SerializeToString())
        data = dict(
            docs=[
                dict(
                    url='snapshot:external-{}-{}'.format(operator, region),
                    options=dict(
                        modification_timestamp=timestamp,
                    ),
                    Operator=[
                        dict(type='#pk', value=operator),
                    ],
                    Region=[
                        dict(type='#pk', value=region),
                    ],
                    Timestamp=[
                        dict(type='#p', value=str(timestamp)),
                    ],
                    Data=[
                        dict(type='#p', value=proto),
                    ],
                )
            ],
            action='modify',
            prefix=0,
        )
        try:
            self._send_document(THIEF_SAAS['prestable_host'], data)
        except Exception as exc:
            LOGGER.exception(exc)
        try:
            self._send_document(THIEF_SAAS['host'], data)
        except Exception as exc:
            LOGGER.exception(exc)

    @retry(RuntimeError, tries=3, delay=2)
    def _send_document(self, host, json_data):
        response = self._session.post(
            'http://{}/service/{}'.format(
                host, THIEF_SAAS['service'],
            ),
            json=json_data,
        )
        if response.status_code != 200:
            raise RuntimeError(
                'Unable to upload data: {}'
                .format(response.content.decode('utf-8'))
            )
