# -*- coding: utf-8 -*-

from requests import get
from retry import retry
from datetime import datetime
from json import loads
from .provider import CarState, Provider
from ..helpers import make_animals_session


class Belkacar(Provider):
    BASE_URL = 'https://api.belkacar.ru/v2.10'

    APP_VERSION = '1.17.03'

    UPDATE_DELAY = 12 * 60 * 60

    def load_car_states(self, account):
        params = {
            'latitude': str(account._geopoint[0]),
            'longitude': str(account._geopoint[1]),
            # Magic constants
            'type_id_list[0]': '1',
            'type_id_list[1]': '2',
            'type_id_list[2]': '5',
            'type_id_list[3]': '38',
        }
        url = '{}/{}'.format(self.BASE_URL, 'car')
        data = self._load_data(account, url, params)
        timestamp = int(datetime.utcnow().timestamp())
        car_states = []
        for car in data['list']:
            tariff = {
                'reserve_price': 0.,
                'park_price': 0.,
                'ride_price': 0.,
            }
            ride_price_by_km = 0.
            data = {
                'number': car['gos_number'],
                'colour': car['colour'],
                'ride_price_by_km': ride_price_by_km,
                'no_refill_distance': car['no_refill_distance'],
                'pins': car['pins'],
                'hardware': car['hardware'],
            }
            for price in car['prices']:
                price['price'] = round(price['price'], 3)
                if price['fare_price_type_id'] == 1:
                    tariff['reserve_price'] = price['price']
                    data['reservation_price'] = price['price']  # Kostyl for SAAS
                elif price['fare_price_type_id'] == 3:
                    tariff['ride_price'] = price['price']
                    data['riding_price'] = price['price']  # Kostyl for SAAS
                elif price['fare_price_type_id'] == 4:
                    ride_price_by_km = price['price']
                elif price['fare_price_type_id'] == 5:
                    tariff['park_price'] = price['price']
                    data['parking_price'] = price['price']  # Kostyl for SAAS
            car_states.append(CarState(
                car_id=str(car['id']),
                timestamp=timestamp,
                region=account.region,
                operator='belkacar',
                position=(car['latitude'], car['longitude']),
                tariffs=[tariff],
                data=data,
            ))
        return car_states

    def update_account(self, account):
        current_time = int(datetime.utcnow().timestamp())
        # If account is almost new, we do not need to refresh auth token
        if account.updated + self.UPDATE_DELAY > current_time:
            return False
        data = {
            'phone': account.data['phone'],
            'device_id': account.data['device_id'],
            'token': account.data['token'],
            'refresh_hash': account.data['refresh_hash'],
        }
        url = '{}/{}'.format(self.BASE_URL, 'auth/refresh-token')
        data = self._post_data(account, url, data)
        account.data['token'] = data['token']
        account.data['refresh_hash'] = data['refresh_hash']
        account.updated = current_time
        return True

    @retry(tries=3)
    def _load_data(self, account, url, params={}):
        response = account._session.get(
            url,
            headers=self._get_headers(account),
            params=params,
        )
        if response.status_code != 200:
            raise Exception()
        return loads(response.content.decode('utf-8'))

    @retry(tries=3)
    def _post_data(self, account, url, data={}):
        response = account._session.post(
            url,
            headers=self._get_headers(account),
            data=data,
        )
        if response.status_code != 200:
            raise Exception()
        return loads(response.content.decode('utf-8'))

    def _get_headers(self, account):
        return {
            'accept': 'application/json',
            'user-agent': 'okhttp/3.11.0',
            '-x-belkacar-auth-token': account.data['token'],
            '-x-belkacar-device-id': account.data['device_id'],
            '-x-belkacar-os': account.data['os'],
            '-x-belkacar-device': account.data['device'],
            '-x-belkacar-appversion': self.APP_VERSION,
            '-x-belkacar-language': 'en',
            '-x-belkacar-user-loc-lat': str(account._geopoint[0]),
            '-x-belkacar-user-loc-lng': str(account._geopoint[1]),
        }
