# -*- coding: utf-8 -*-

from requests import get
from retry import retry
from datetime import datetime
from json import loads
from .provider import CarState, Provider


class Carsmile(Provider):
    BASE_URL = 'https://api.carsmile.com'

    def load_car_states(self, account):
        data = self._load_data(
            account,
            """{
                availableCars {
                    id
                    transmission
                    registrationPlate
                    fuelTankVolume
                    currentFuelLevel
                    model {
                        id
                        name
                        brand {
                            id
                            name
                        }
                    }
                    coords {
                        latitude
                        longitude
                    }
                }
            }"""
        )
        timestamp = int(datetime.utcnow().timestamp())
        car_states = []
        for car in data['data']['availableCars']:
            car_id = str(car['id'])
            position = (
                car['coords']['latitude'],
                car['coords']['longitude']
            )
            del car['id']
            del car['coords']
            if 'registrationPlate' in car:
                car['number'] = car['registrationPlate']
            if 'currentFuelLevel' in car:
                car['fuel_level'] = car['currentFuelLevel']
            if 'model' in car:
                car['model_dict'] = car['model']
                car['model'] = '{} {}'.format(
                    car['model_dict']['brand']['name'],
                    car['model_dict']['name']
                )
            car_states.append(CarState(
                car_id=car_id,
                timestamp=timestamp,
                region=account.region,
                operator='carsmile',
                position=position,
                tariffs=[],
                data=car,
            ))
        return car_states

    @retry(tries=3)
    def _load_data(self, account, query):
        headers = {
            'accept': 'application/json',
            'user-agent': 'okhttp/3.11.0',
            'Content-Type': 'application/json',
        }
        response = account._session.post(
            self.BASE_URL,
            json={
                'query': query,
            },
            headers=headers,
        )
        if response.status_code != 200:
            raise Exception()
        return loads(response.content.decode('utf-8'))
