# -*- coding: utf-8 -*-

from requests import get
from retry import retry
from datetime import datetime
from json import loads
from .provider import CarState, Provider
from ..helpers import make_animals_session


class YandexDrive(Provider):
    BASE_URL = 'http://prestable.carsharing.yandex.net'

    APP_BUILD = 100000

    def load_car_states(self, account):
        url = '{}/{}'.format(self.BASE_URL, 'external/car/list')
        data = self._load_data(account, url)
        timestamp = int(datetime.utcnow().timestamp())
        car_states = []
        for car in data['cars']:
            car_id = car['number']
            position = (car['location']['lat'], car['location']['lon'])
            del car['number']
            del car['location']
            car_states.append(CarState(
                car_id=car_id,
                timestamp=timestamp,
                region=self._detect_region(position),
                operator='yandexdrive',
                position=position,
                tariffs=[],
                data=car,
            ))
        return car_states

    @staticmethod
    def _detect_region(position):
        to_msk   = (position[0] - 55.755786)**2 + (position[1] - 37.622332)**2
        to_spb   = (position[0] - 59.934159)**2 + (position[1] - 30.337418)**2
        to_kazan = (position[0] - 55.830430)**2 + (position[1] - 49.066081)**2
        if to_msk < to_spb:
            return 'kazan' if to_kazan < to_msk else 'msk'
        else:
            return 'kazan' if to_kazan < to_spb else 'spb'

    @retry(tries=3)
    def _load_data(self, account, url, params={}):
        response = account._session.get(
            url,
            headers=self._get_headers(account),
        )
        if response.status_code != 200:
            raise Exception()
        return loads(response.content.decode('utf-8'))

    def _get_headers(self, account):
        return {
            'Accept': 'application/json',
            'User-Agent': 'okhttp/3.11.0',
            'Authorization': 'OAuth {}'.format(account.data['token']),
            'AppBuild': str(self.APP_BUILD)
        }
