# -*- coding: utf-8 -*-

from requests import get
from retry import retry
from random import uniform
from datetime import datetime
from json import loads
from copy import deepcopy
from .provider import CarState, Provider
from ..helpers import make_animals_session


class Youdrive(Provider):
    BASE_URL = 'https://cs-v2.youdrive.today'

    API_VERSION = '7'

    def load_car_states(self, account):
        params = {
            # 'lat': 55.751244 + uniform(-0.0001, 0.0001),
            # 'lon': 37.618423 + uniform(-0.0001, 0.0001),
            'version': self.API_VERSION,
        }
        url = '{}/{}'.format(self.BASE_URL, 'status')
        data = self._load_data(account, url, params)
        timestamp = int(datetime.utcnow().timestamp())
        car_states = []
        for car in data['cars']:
            car_id = car['car_id']
            position = (car['lat'], car['lon'])
            tariff = deepcopy(car['tariff'])
            del car['car_id']
            del car['lat']
            del car['lon']
            del car['tariff']
            if 'fuel' in car:
                car['fuel_level'] = car['fuel']
            if 'parking' in tariff:
                car['parking_price'] = tariff['parking']['price'] / 100
            if 'usage' in tariff:
                car['riding_price'] = tariff['usage']['price'] / 100
            car_states.append(CarState(
                car_id=car_id,
                timestamp=timestamp,
                region=self._detect_region(position),
                operator='youdrive',
                position=position,
                tariffs=[tariff],
                data=car
            ))
        return car_states

    @staticmethod
    def _detect_region(position):
        to_msk   = (position[0] - 55.755786)**2 + (position[1] - 37.622332)**2
        to_spb   = (position[0] - 59.934159)**2 + (position[1] - 30.337418)**2
        to_kazan = (position[0] - 55.830430)**2 + (position[1] - 49.066081)**2
        if to_msk < to_spb:
            return 'kazan' if to_kazan < to_msk else 'msk'
        else:
            return 'kazan' if to_kazan < to_spb else 'spb'

    def _load_data(self, account, url, params={}):
        response = account._session.get(
            url,
            headers=self._get_headers(account),
            cookies=self._get_cookies(account),
            params=params,
        )
        if response.status_code != 200:
            raise Exception(
                response.status_code,
                response.content.decode('utf-8')
            )
        return loads(response.content.decode('utf-8'))

    def _get_headers(self, account):
        return {
            'Accept-Encoding': 'gzip',
            'Accept-Language': 'en',
            'User-Agent': 'carsharing/3.0.153 (Linux; Android 6.0.1; SM-G532G Build/MMB29T)',
        }

    def _get_cookies(self, account):
        return {
            'session_id': account.data['session_id']
        }
