import retry

import django.db
from django.db import transaction
from django.utils import timezone
import psycopg2

from ..models.app_install import AppInstall
from ..models.user import User


class AppInstallUpdater(object):
    db_exceptions = (
        django.db.DatabaseError,
        django.db.OperationalError,
        psycopg2.DatabaseError,
        psycopg2.OperationalError,
    )

    def __init__(self, user, push_client=None):
        self._user = user
        self._push_client = push_client

    # try to re-apply transaction if lock fails / PgBouncer connection errors
    @retry.retry(exceptions=db_exceptions, tries=3, delay=0.001, backoff=2)
    def update(self, uuid, device_id, platform, app_name, app_version, app_build, push_token):
        now = timezone.now()

        try:
            app_install = self._update_atomic(
                uuid, device_id, platform, app_name, app_build, app_version, push_token, now
            )
        except self.db_exceptions:
            django.db.close_old_connections()
            raise

        return app_install

    @transaction.atomic(savepoint=False)
    def _update_atomic(self, uuid, device_id, platform, app_name, app_build, app_version, push_token, now):
        latest_app_install = self._lock_latest_app_install()
        if latest_app_install is None:
            self._lock_user()

        # pylint: disable=unused-argument
        app_installs = list(AppInstall.objects.filter(user=self._user))
        app_install = self._find_app_install(candidates=app_installs, uuid=uuid)
        if app_install is None:
            app_install = (
                AppInstall.objects
                .create(
                    user=self._user,
                    uuid=uuid,
                    device_id=device_id,
                    platform=platform.value,
                    app_name=app_name,
                    app_version=app_version,
                    app_build=app_build,
                    push_token=push_token,
                    is_latest=True,
                    created_at=now,
                    refreshed_at=now,
                )
            )
            self._log_new_app_install(
                new_app_install=app_install,
                prev_app_install=latest_app_install,
            )
            if latest_app_install:
                latest_app_install.is_latest = False
                latest_app_install.save()
        else:
            if push_token and push_token != app_install.push_token:
                app_install.push_token = push_token

            if not app_install.is_latest:
                app_install.is_latest = True
                latest_app_install.is_latest = False
                latest_app_install.save()

            if app_build:
                app_install.app_build = app_build
            if app_version:
                app_install.app_version = app_version

            app_install.refreshed_at = now

            app_install.save()

        return app_install

    def _lock_latest_app_install(self):
        try:
            latest_app_install = (
                AppInstall.objects
                .select_for_update()
                .get(user=self._user, is_latest=True)
            )
        except AppInstall.DoesNotExist:
            latest_app_install = None
        return latest_app_install

    def _lock_user(self):
        User.objects.select_for_update().get(id=self._user.id)

    def _find_app_install(self, candidates, uuid):
        app_install = None
        for candidate in candidates:
            if candidate.uuid == uuid:
                app_install = candidate
                break
        return app_install

    def _mark_latest_app_install(self, app_install):
        pass

    def _log_new_app_install(self, new_app_install, prev_app_install):
        if self._push_client is None:
            return

        new_app_install_obj = self._app_install_to_json_compatible_object(new_app_install)
        if prev_app_install is None:
            prev_app_install_obj = None
        else:
            prev_app_install_obj = self._app_install_to_json_compatible_object(prev_app_install)

        data = {
            'new_app_install': new_app_install_obj,
            'prev_app_install': prev_app_install_obj,
        }

        self._push_client.log(type_='app_install.new', data=data)

    def _app_install_to_json_compatible_object(self, app_install):
        return {
            'id': str(app_install.id),
            'user_id': str(app_install.user_id),
            'uuid': str(app_install.uuid),
            'device_id': str(app_install.device_id),
            'platform': app_install.platform,
            'push_token': app_install.push_token,
            'created_at': app_install.created_at.timestamp(),
            'refreshed_at': app_install.refreshed_at.timestamp(),
            'app_name': app_install.app_name,
            'app_version': app_install.app_version,
            'app_build': app_install.app_build,
        }
