import logging
import warnings

import cars.settings
from cars.core.datasync import DataSyncClient
from cars.core.push_client import CarsharingPushClientLogger
from cars.core.tvm2 import TVM2ServiceTicket
from cars.django.lock import UwsgiLock


LOGGER = logging.getLogger(__name__)


class DataSyncDocumentsClient:

    PASSPORT_COLLECTION = 'documents'
    LICENSE_COLLECTION = 'driving_license'
    PASSPORT_UNVERIFIED_COLLECTION = 'documents_unverified'
    LICENSE_UNVERIFIED_COLLECTION = 'driving_license_unverified'

    VERIFIED_NAME = 'carsharing'

    service_ticket = TVM2ServiceTicket(**cars.settings.USERS['documents']['datasync']['tvm2'])

    class EmptyDocumentError(Exception):
        pass

    NotFoundError = DataSyncClient.NotFoundError

    def __init__(self, datasync_client, passport_client=None):
        self._client = datasync_client
        self._passport_client = passport_client

    @classmethod
    def from_settings(cls, passport_client=None):
        push_client = CarsharingPushClientLogger(
            filename=cars.settings.USERS['documents']['datasync']['push_client']['filename'],
            lock=UwsgiLock(),
        )
        datasync_client = DataSyncClient(
            host=cars.settings.USERS['documents']['datasync']['host'],
            endpoint_path=cars.settings.USERS['documents']['datasync']['endpoints_path'],
            service_ticket=cls.service_ticket,
            push_client=push_client,
            verify_requests=cars.settings.USERS['documents']['datasync']['verify_requests'],
        )
        client = cls(datasync_client=datasync_client, passport_client=passport_client)
        return client

    # Passport unverified

    def get_passport_unverified(self, uid, key):
        return self._get_user_data(self.PASSPORT_UNVERIFIED_COLLECTION, uid, key)

    def update_passport_unverified(self, uid, key, new_data):
        return self._update_user_data(self.PASSPORT_UNVERIFIED_COLLECTION, uid, key, new_data)

    # License unverified

    def get_license_unverified(self, uid, key):
        license_data = self._get_user_data(self.LICENSE_UNVERIFIED_COLLECTION, uid, key)
        if not license_data:
            return license_data
        license_data['number'] = license_data.get('number_front') or license_data.get('number_back')
        license_data['categories'] = 'B'
        return license_data

    def update_license_unverified(self, uid, key, new_data):
        return self._update_user_data(self.LICENSE_UNVERIFIED_COLLECTION, uid, key, new_data)

    # Verified passport

    def get_passport(self, uid):
        warnings.warn(
            'We\'re moving towards one DS collection per one document type.'
            'Please use unverified one and method get_passport_unverified '
            'with user.passport_ds_revision instead.',
            DeprecationWarning
        )

        return self._get_user_data(self.PASSPORT_COLLECTION, uid, self.VERIFIED_NAME)

    def update_passport(self, uid, new_data):
        return self._update_user_data(self.PASSPORT_COLLECTION, uid, self.VERIFIED_NAME, new_data)

    # Verified license

    def get_license(self, uid):
        warnings.warn(
            'We\'re moving towards one DS collection per one document type.'
            'Please use unverified one and method get_license_unverified '
            'with user.driving_license_ds_revision instead.',
            DeprecationWarning
        )
        return self._get_user_data(self.LICENSE_COLLECTION, uid, self.VERIFIED_NAME)

    def update_license(self, uid, new_data):
        return self._update_user_data(self.LICENSE_COLLECTION, uid, self.VERIFIED_NAME, new_data)

    # Move to verified

    def copy_passport_to_verified(self, uid, key):
        passport_data = self.get_passport_unverified(uid, key)
        if not passport_data:
            raise DataSyncDocumentsClient.NotFoundError

        self.update_passport(uid, passport_data)

    def copy_license_to_verified(self, uid, key):
        license_data = self.get_license_unverified(uid, key)
        if not license_data:
            raise DataSyncDocumentsClient.NotFoundError

        license_data['number'] = license_data['number_front'] or license_data['number_back']
        license_data['categories'] = 'B'

        if 'number_front' in license_data:
            del license_data['number_front']
        if 'number_back' in license_data:
            del license_data['number_back']

        self.update_license(uid, license_data)

    # Helpers

    def _get_user_data(self, collection, uid, key):
        uid = abs(int(uid))
        try:
            data = self._client.get(collection, str(uid), str(key))
        except self._client.NotFoundError:
            data = None
        except Exception:
            if self._passport_client is None:
                raise

            do_account_exists = self._passport_client.do_account_exists(str(uid), default=True)
            if do_account_exists:
                raise

            LOGGER.exception(
                'cannot obtain {} user data (uid = {}) with key = {} from datasync:'
                ' user does not exist in blackbox'
                .format(collection, uid, key)
            )
            data = None

        return data

    def _update_user_data(self, collection, uid, key, new_data):
        uid = abs(int(uid))
        if not new_data:
            raise self.EmptyDocumentError

        new_data['id'] = str(key)
        return self._client.put(collection, str(uid), new_data)
