import base64
import datetime
import logging
from abc import abstractmethod

import pytz
from django.utils import timezone


LOGGER = logging.getLogger(__name__)


class BaseConverter:

    @abstractmethod
    def convert(self, data):
        raise NotImplementedError


class BaseSmartIDConverter(BaseConverter):  # pylint:disable=abstract-method

    def _parse_date_from_scanners_list(self, scanners_lst, raw_data_name):
        for scanner in scanners_lst:
            parsed = self._parse_date(scanner.fields, raw_data_name)
            if parsed is not None:
                return parsed
        return None

    def _parse_string_from_scanners_list(self, scanners_lst, raw_data_name):
        for scanner in scanners_lst:
            parsed = self._parse_string(scanner.fields, raw_data_name)
            if parsed is not None:
                return parsed
        return None

    def _parse_date(self, raw_data, raw_data_name):
        date_obj = raw_data.get(raw_data_name)
        if date_obj is None:
            return None
        date_str = date_obj.value
        try:
            date_dt = datetime.datetime.strptime(date_str, '%d.%m.%Y')
        except ValueError:
            LOGGER.exception('Unable to parse date.')
            return None
        date_dt = timezone.make_aware(date_dt, timezone=pytz.UTC)
        return date_dt.isoformat()

    def _parse_string(self, raw_data, raw_data_name):
        if raw_data_name in raw_data:
            return raw_data[raw_data_name].value
        return None


class UserDrivingLicenseSmartIDConverter(BaseSmartIDConverter):

    def convert(self, data):
        pre_payload = {
            'id': 'initial',
            'number': self._parse_number(data['front']),
            'first_name': self._parse_string_from_scanners_list(data['front'], 'name_rus'),
            'last_name': self._parse_string_from_scanners_list(data['front'], 'surname_rus'),
            'middle_name': self._parse_string_from_scanners_list(data['front'], 'patronymic_rus'),
            'issue_date': self._parse_date_from_scanners_list(data['front'], 'issue_date'),
            'birth_date': self._parse_date_from_scanners_list(data['front'], 'birth_date'),
        }
        result = {
            field_name: field_value
            for field_name, field_value in pre_payload.items()
            if field_value is not None
        }
        return result

    def _parse_number(self, obj):
        parsed_number = self._parse_string_from_scanners_list(
            scanners_lst=obj,
            raw_data_name='number',
        )
        if not parsed_number:
            return None
        return parsed_number.replace(' ', '')


class UserPassportSmartIDConverter(BaseSmartIDConverter):

    def convert(self, data):
        pre_payload = {
            'id': 'initial',
            'doc_type': 'id',
            'citizenship': 'РОССИЙСКАЯ ФЕДЕРАЦИЯ',
            'doc_value': self._parse_doc_value(data['biographical']),
            'first_name': self._parse_string_from_scanners_list(
                data['biographical'],
                'name'
            ),
            'last_name': self._parse_string_from_scanners_list(
                data['biographical'],
                'surname'
            ),
            'middle_name': self._parse_string_from_scanners_list(
                data['biographical'],
                'patronymic'
            ),
            'issue_date': self._parse_date_from_scanners_list(data['biographical'], 'issue_date'),
            'birth_date': self._parse_date_from_scanners_list(data['biographical'], 'birthdate'),
            'birth_place': self._parse_string_from_scanners_list(
                data['biographical'],
                'birthplace'
            ),
            'subdivision_code': self._parse_string_from_scanners_list(
                data['biographical'],
                'authority_code'
            ),
            'gender': self._parse_gender(data['biographical']),
        }
        result = {
            field_name: field_value
            for field_name, field_value in pre_payload.items()
            if field_value is not None
        }
        return result

    def _parse_doc_value(self, obj):
        for scanner in obj:
            data = scanner.fields
            if 'series' in data and 'number' in data:
                return data['series'].value + data['number'].value
        return None

    def _parse_gender(self, obj):
        raw_gender = self._parse_string_from_scanners_list(obj, 'gender')
        if raw_gender is None:
            return None
        return raw_gender[:3].rstrip('.')


class UserDrivingLicensePDF417Converter(BaseConverter):

    def convert_from_binary_data(self, binary_data):
        decoded_data = None

        for encoding in ['utf-8', 'cp1251', 'koi8-r']:
            try:
                decoded_data = binary_data.decode(encoding)
                break
            except UnicodeDecodeError:
                LOGGER.exception(
                    'failed to decode driver license barcode with %s encoding',
                    encoding,
                )
                continue

        if not decoded_data:
            return {}

        try:
            decoded_data = base64.b64decode(decoded_data).decode('utf-8')
        except ValueError:
            pass

        tokens = decoded_data.split('|')
        result = {
            'number': tokens[0],
            'last_name': tokens[3],
            'first_name': tokens[4],
            'middle_name': tokens[5],
        }
        if self._parse_date(tokens[1]):
            result['categories_b_valid_from_date'] = self._parse_date(tokens[1])
        if self._parse_date(tokens[2]):
            result['categories_b_valid_to_date'] = self._parse_date(tokens[2])
        if self._parse_date(tokens[6]):
            result['birth_date'] = self._parse_date(tokens[6])
        return result

    def convert(self, data):
        if 'back' not in data:
            return {}

        result = {}
        for scanner in data['back']:
            if 'data' not in scanner.fields:
                continue

            binary_data = base64.b64decode(scanner.fields['data'].value)
            converted_scanner_data = self.convert_from_binary_data(binary_data)
            result.update(converted_scanner_data)

        return result

    def _parse_date(self, date_str):
        try:
            date_dt = datetime.datetime.strptime(date_str, '%Y%m%d')
        except ValueError:
            return None
        date_dt = timezone.make_aware(date_dt, timezone=pytz.UTC)
        return date_dt.isoformat()
