import copy
import datetime
import logging
import time

from django.db import transaction
from django.utils import timezone

import kubiki.passport

import cars.settings
from cars.core.solomon import make_solomon_client
from .user_profile_updater import UserProfileUpdater
from ..models import User, UserPhoneBinding


LOGGER = logging.getLogger(__name__)


class UserPhoneBinder(object):

    class NotSubmittedError(Exception):
        pass

    class PhoneExists(Exception):
        pass

    class RecentSubmissionError(Exception):
        pass

    PassportError = kubiki.passport.PassportError

    def __init__(self, passport_client, solomon_client):
        self._min_resubmit_interval = datetime.timedelta(seconds=25)
        self._client = passport_client
        self._solomon = solomon_client

    @classmethod
    def from_settings(cls):
        passport_settings = cars.settings.PASSPORT

        solomon_config = copy.deepcopy(cars.settings.SOLOMON)
        solomon_config.update(cars.settings.COMMON['solomon'])
        solomon_client = make_solomon_client(config=solomon_config)

        passport_client = kubiki.passport.Passport(
            consumer=passport_settings['consumer'],
            url=passport_settings['url'],
            timeout=passport_settings['timeout'],
            verify_ssl=passport_settings['verify_ssl'],
            session_params=passport_settings['session_params'],
        )

        return cls(
            passport_client=passport_client,
            solomon_client=solomon_client,
        )

    def bind_submit(self, user, user_ip, user_ua, number, display_language, oauth_token):
        with transaction.atomic(savepoint=False):
            # Write-lock the user.
            User.objects.select_for_update().get(id=user.id)
            return self._do_bind_submit(
                user=user,
                user_ip=user_ip,
                user_ua=user_ua,
                number=number,
                display_language=display_language,
                oauth_token=oauth_token,
            )

    def _do_bind_submit(self, user, user_ip, user_ua, number, display_language, oauth_token):
        latest_binding = self._get_latest_binding(user=user)
        if (latest_binding is not None
                and timezone.now() - latest_binding.submit_date < self._min_resubmit_interval):
            raise self.RecentSubmissionError

        start_time = time.time()

        response = self._client.phone_bind_submit(
            client_ip=user_ip,
            client_scheme='https',
            client_user_agent=user_ua,
            number=number,
            display_language=display_language,
            oauth_token=oauth_token,
        )

        self._solomon.set_value(
            'phone_binder.submit.response.time',
            (time.time() - start_time) * 1000,
            labels={
                'status_code': response.raw_response.status_code,
            },
        )

        response.raise_for_status()

        normalized_number = response.data['number']['e164']
        track_id = response.data['track_id']

        UserPhoneBinding.objects.create(
            user=user,
            phone=normalized_number,
            track_id=track_id,
            submit_date=timezone.now(),
        )

    def bind_commit(self, user, user_ip, user_ua, code, oauth_token):
        with transaction.atomic(savepoint=False):
            # Write-lock binding object to avoid concurrent upates.
            phone_binding = self._get_latest_binding(user=user, lock=True)
            if phone_binding is None:
                raise self.NotSubmittedError

            # Concurrency again.
            if phone_binding.commit_date is not None:
                assert user.phone == phone_binding.phone, \
                    '{} != {}'.format(user.phone, phone_binding.phone)
                return

            start_time = time.time()

            response = self._client.phone_bind_commit(
                client_ip=user_ip,
                client_scheme='https',
                client_user_agent=user_ua,
                code=code,
                track_id=phone_binding.track_id,
                oauth_token=oauth_token,
            )

            self._solomon.set_value(
                'phone_binder.commit.response.time',
                (time.time() - start_time) * 1000,
                labels={
                    'status_code': response.raw_response.status_code,
                },
            )

            response.raise_for_status()

            normalized_number = response.data['number']['e164']
            assert phone_binding.phone == normalized_number, \
                '{} != {}'.format(phone_binding.phone, normalized_number)

            # If another user has the same phone number, set it to unverified
            try:
                conflict_user = User.objects.get(
                    phone=phone_binding.phone,
                    is_phone_verified=True,
                )
            except User.DoesNotExist:
                # Normal scenario
                pass
            else:
                UserProfileUpdater(conflict_user).request_phone_verification()

            UserProfileUpdater(user).update_phone(phone_binding.phone)

            phone_binding.commit_date = timezone.now()
            phone_binding.save()

    def _get_latest_binding(self, user, lock=False):
        qs = UserPhoneBinding.objects
        if lock:
            qs = qs.select_for_update()
        phone_binding = (
            qs
            .filter(user=user)
            .order_by('-submit_date')
            .first()
        )
        return phone_binding
