import binascii
import io
import base64
import datetime
import hmac
import logging
import tempfile
import time
from hashlib import sha1

import ffprobe3
import ffmpy

from django.db import transaction
from django.db.models import Q
from django.utils import timezone
from django.core.exceptions import ValidationError
from django.core.validators import EmailValidator

import cars.settings
from cars.core.new_billing import NewBillingClient
from cars.core.history import HistoryManager
from cars.core.saas_drive_admin import SaasDriveAdminClient
from ...core.constants import AppPlatform
from ..models import (
    User, UserCreditCard, UserCreditCardHistory,
    UserDocument, UserDocumentBackgroundVideo, UserDocumentPhoto,
    UserPersdataProcessingConsent, UserDataHistory
)
from .documents_textdata_converter import UserDrivingLicensePDF417Converter
from ..core.datasync import DataSyncDocumentsClient

LOGGER = logging.getLogger(__name__)


class UserHistoryManager(HistoryManager):
    def __init__(self):
        super().__init__(
            UserDataHistory,
            excluded_fields=('password', ),
            coercing_rules={'tags': (lambda x: (str(x)[:128] if x is not None else None))}
        )
        self._robot_id = self._init_robot_id()

    def _init_robot_id(self):
        return 'robot-carsharing'

    def add_entry(self, new_object_state, operator_id=None, timestamp_override=None):
        operator_id = str(operator_id) if operator_id is not None else self._robot_id
        super().add_entry(new_object_state, operator_id, timestamp_override)

    def update_entry(self, new_object_state, operator_id=None, timestamp_override=None):
        operator_id = str(operator_id) if operator_id is not None else self._robot_id
        super().update_entry(new_object_state, operator_id, timestamp_override)

    def remove_entry(self, new_object_state, operator_id=None, timestamp_override=None):
        operator_id = str(operator_id) if operator_id is not None else self._robot_id
        super().remove_entry(new_object_state, operator_id, timestamp_override)


class UserProfileUpdater(object):

    user_first_name_field_validator = User._meta.get_field('first_name')

    class ConflictError(Exception):
        pass

    class CreditCardNotBound(Exception):
        pass

    class StatusChangeError(Exception):
        pass

    class ValidationError(Exception):
        pass

    def __init__(self,
                 user,
                 mds_client=None,
                 trust_client=None,
                 recognized_data_submitter=None,
                 push_client=None,
                 datasync_client=DataSyncDocumentsClient.from_settings(),
                 saas_client=SaasDriveAdminClient.from_settings(),
                 new_billing_client=NewBillingClient.from_settings()):
        self._user = user
        self._mds_client = mds_client
        self._trust_client = trust_client
        self._recognized_data_submitter = recognized_data_submitter
        self._push_client = push_client
        self._datasync_client = datasync_client
        self._saas_client = saas_client
        self._new_billing_client = new_billing_client
        self._history_manager = UserHistoryManager()

    def __repr__(self):
        return '<UserProfileUpdater: user.id={}>'.format(self._user.id)

    def update_username(self, username, performer_id=None):
        # used in migration of social accounts to ordinary ones
        prev_username = self._user.username
        self._user.username = username
        self._user.updated_at = time.time()

        with transaction.atomic(savepoint=False):
            self._user.save()
            self._history_manager.update_entry(self._user, performer_id)

        self._log(
            subtype='username',
            data={
                'user_id': str(self._user.id),
                'old': prev_username,
                'new': username,
            },
        )

    def update_is_yandexoid(self, is_yandexoid, performer_id=None):
        prev_is_yandexoid = self._user.is_yandexoid
        self._user.is_yandexoid = is_yandexoid
        self._user.updated_at = time.time()

        with transaction.atomic(savepoint=False):
            self._user.save()
            self._history_manager.update_entry(self._user, performer_id)

        self._log(
            subtype='is_yandexoid',
            data={
                'user_id': str(self._user.id),
                'old': prev_is_yandexoid,
                'new': self._user.is_yandexoid,
            },
        )

    def update_registered_at(self, registered_at, performer_id=None):
        prev_registered_at = self._user.registered_at
        self._user.registered_at = registered_at
        self._user.updated_at = time.time()

        self._user.save()
        self._history_manager.update_entry(self._user, performer_id)

        self._log(
            subtype='registered_at',
            data={
                'user_id': str(self._user.id),
                'old': prev_registered_at.isoformat() if prev_registered_at else None,
                'new': self._user.registered_at.isoformat(),
            },
        )

    def update_status(self, status, force=False, performer_id=None):
        prev_status = User.Status(self._user.status)

        if not force and not self._can_change_status(prev_status, status):
            raise self.StatusChangeError()

        self._user.status = status.value
        self._user.updated_at = time.time()

        with transaction.atomic(savepoint=False):
            self._user.save()
            self._history_manager.update_entry(self._user, performer_id)

        self._log(
            subtype='status',
            data={
                'user_id': str(self._user.id),
                'old': prev_status,
                'new': self._user.status,
            },
        )

    def update_persdata_processing_consent(self, consent_type, consent_value, performer_id=None):
        with transaction.atomic(savepoint=False):
            consents = self._user.persdata_consents
            if consents is None:
                self._user.persdata_consents = UserPersdataProcessingConsent()
                self._user.persdata_consents.save()
                self._user.updated_at = time.time()
                self._user.save()

            if consent_type == 'porsche':
                self._user.persdata_consents.porsche_consent = consent_value
                self._user.persdata_consents.porsche_consent_updated_at = time.time()
                self._user.persdata_consents.save()

                self._user.updated_at = time.time()
                self._user.save()
            else:
                LOGGER.info('consent type %s is unreachable', consent_type)

            self._history_manager.update_entry(self._user, performer_id)

    def update_email(self, email, performer_id=None, use_proxy=False):
        try:
            EmailValidator()(email)
        except ValidationError:
            raise self.ValidationError

        try:
            existing_user = User.objects.get(email=email)
        except User.DoesNotExist:
            existing_user = None
        except User.MultipleObjectsReturned:
            raise self.ConflictError

        if existing_user is not None and existing_user != self._user:
            raise self.ConflictError

        prev_email = self._user.email

        if use_proxy:
            try:
                self._saas_client.update_email(self._user.id, email)
            except Exception:
                self._update_email_internal(email, performer_id)
            else:
                self._user.refresh_from_db()
        else:
            self._update_email_internal(email, performer_id)

        self._log(
            subtype='email',
            data={
                'user_id': str(self._user.id),
                'old': prev_email,
                'new': self._user.email,
            },
        )

    def _update_email_internal(self, email, performer_id):
        self._user.email = email
        self._user.is_email_verified = True
        self._user.updated_at = time.time()

        with transaction.atomic(savepoint=False):
            self._user.save()
            self._history_manager.update_entry(self._user, performer_id)

    def update_first_name(self, first_name, commit=True, performer_id=None):
        if first_name is not None and len(first_name) > 30:
            first_name = first_name[:30]
        prev_first_name = self._user.first_name
        curr_first_name = first_name.strip().capitalize()

        try:
            self._user.first_name = self.user_first_name_field_validator.clean(curr_first_name, None)
        except ValidationError:
            LOGGER.error('invalid first name "{}"'.format(first_name.encode('utf-8')))
            raise

        if commit:
            self._user.updated_at = time.time()

            with transaction.atomic(savepoint=False):
                self._user.save()
                self._history_manager.update_entry(self._user, performer_id)

        self._log(
            subtype='first_name',
            data={
                'user_id': str(self._user.id),
                'old': prev_first_name,
                'new': self._user.first_name,
            },
        )

    def update_last_name(self, last_name, commit=True, performer_id=None):
        prev_last_name = self._user.last_name
        self._user.last_name = last_name.strip().capitalize()

        if commit:
            self._user.updated_at = time.time()

            with transaction.atomic(savepoint=False):
                self._user.save()
                self._history_manager.update_entry(self._user, performer_id)

        self._log(
            subtype='last_name',
            data={
                'user_id': str(self._user.id),
                'old': prev_last_name,
                'new': self._user.last_name,
            },
        )

    def update_patronymic_name(self, patronymic_name, commit=True, performer_id=None):
        prev_patronymic_name = self._user.patronymic_name
        self._user.patronymic_name = patronymic_name.strip().capitalize()

        if commit:
            self._user.updated_at = time.time()

            with transaction.atomic(savepoint=False):
                self._user.save()
                self._history_manager.update_entry(self._user, performer_id)

        self._log(
            subtype='patronymic_name',
            data={
                'user_id': str(self._user.id),
                'old': prev_patronymic_name,
                'new': self._user.patronymic_name,
            },
        )

    def update_passport(self, passport_number, first_name, last_name, patronymic_name):
        passport = (
            UserDocument.objects
            .filter(
                user=self._user,
                type=UserDocument.Type.PASSPORT.value,
            )
            .order_by('-verified_at')
            .first()
        )

        if first_name or last_name or patronymic_name:
            self.update_first_name(first_name, commit=False)
            self.update_last_name(last_name, commit=False)
            self.update_patronymic_name(patronymic_name, commit=True)

    def update_driver_license(self, driver_license_number, first_name, last_name, patronymic_name):
        driver_license = (
            UserDocument.objects
            .filter(
                user=self._user,
                type=UserDocument.Type.DRIVER_LICENSE.value,
            )
            .order_by('-verified_at')
            .first()
        )

        if first_name or last_name or patronymic_name:
            self.update_first_name(first_name, commit=False)
            self.update_last_name(last_name, commit=False)
            self.update_patronymic_name(patronymic_name, commit=True)

    def update_unverified_driver_license(self,
                                         front_content, front_recognized,
                                         back_content, back_recognized):
        try:
            # Don't let any kind of error disrupt the procedure
            self._recognized_data_submitter.submit_recognized_driver_license(
                self._user.uid,
                front=front_recognized,
                back=back_recognized,
            )
        except Exception:
            LOGGER.exception('Exception while updating recognized driver license data.')

        front_photo = self.update_unverified_driver_license_front(
            content=front_content,
            recognized=front_recognized,
        )
        back_photo = self.update_unverified_driver_license_back(
            content=back_content,
            recognized=back_recognized,
        )
        return front_photo, back_photo

    def update_unverified_driver_license_front(self, content, recognized):
        photo = self._update_document_photo_content(
            document_type=UserDocument.Type.DRIVER_LICENSE,
            photo_type=UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT,
            content=content,
        )
        _ = recognized
        return photo

    def update_unverified_driver_license_back(self, content, recognized):
        photo = self._update_document_photo_content(
            document_type=UserDocument.Type.DRIVER_LICENSE,
            photo_type=UserDocumentPhoto.Type.DRIVER_LICENSE_BACK,
            content=content,
        )

        for recognized_document in recognized:
            if recognized_document.scanner.lower().startswith('pdf417'):
                binary_barcode = recognized_document.fields.get('data')
                if binary_barcode is None:
                    LOGGER.info(
                        'missing data from barcode fields: %s',
                        recognized_document.fields.keys(),
                    )
                    continue
                self._update_user_profile_from_driver_license_barcode(
                    b64_barcode=binary_barcode.value,
                )
            else:
                LOGGER.error('unknown scanner: %s', recognized_document.scanner)
                continue

        return photo

    def update_unverified_passport(self,
                                   biographical_content, biographical_recognized,
                                   registration_content, registration_recognized):
        try:
            # Don't let any kind of error disrupt the procedure.
            self._recognized_data_submitter.submit_recognized_passport(
                self._user.uid,
                biographical=biographical_recognized,
                registration=registration_recognized,
            )
        except Exception:
            LOGGER.exception('Exception while updating recognized passport data.')

        biographical_photo = self.update_unverified_passport_biographical(
            content=biographical_content, recognized=biographical_recognized,
        )
        registration_photo = self.update_unverified_passport_registration(
            content=registration_content, recognized=registration_recognized,
        )
        return biographical_photo, registration_photo

    def update_unverified_passport_biographical(self, content, recognized):
        photo = self._update_document_photo_content(
            document_type=UserDocument.Type.PASSPORT,
            photo_type=UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL,
            content=content,
        )
        _ = recognized
        return photo

    def update_unverified_passport_registration(self, content, recognized):
        photo = self._update_document_photo_content(
            document_type=UserDocument.Type.PASSPORT,
            photo_type=UserDocumentPhoto.Type.PASSPORT_REGISTRATION,
            content=content,
        )
        _ = recognized
        return photo

    def update_unverified_passport_selfie(self, selfie_content, face_content=None):
        selfie_photo = self._update_document_photo_content(
            document_type=UserDocument.Type.PASSPORT,
            photo_type=UserDocumentPhoto.Type.PASSPORT_SELFIE,
            content=selfie_content,
        )
        if face_content:
            face_photo = self._update_document_photo_content(
                document_type=UserDocument.Type.PASSPORT,
                photo_type=UserDocumentPhoto.Type.FACE,
                content=face_content,
            )
        else:
            face_photo = None
        return face_photo, selfie_photo

    def convert_x264(self, video_stream, photo_id):
        with tempfile.NamedTemporaryFile(mode='w+b') as infile, \
                tempfile.NamedTemporaryFile(mode='rb') as outfile:
            infile.write(video_stream.read())
            infile.seek(0)
            codec_name = None
            try:
                codec_name = ffprobe3.FFProbe(infile.name).video[0].codec_name
            except Exception:
                LOGGER.exception(
                    'failed ffprobe3 check for photo_id %s',
                    photo_id
                )
            if codec_name == 'h264':  # already has right format
                outfile = infile
            else:
                try:
                    ff_cmd = ffmpy.FFmpeg(inputs={infile.name: None},
                                          outputs={outfile.name: '-y -vcodec libx264 -f mp4'})
                    ff_cmd.run(stdout=None, stderr=None)
                except Exception:
                    LOGGER.exception(
                        'failed to convert android video file to h.264 for photo_id %s',
                        photo_id
                    )
                    outfile = infile  # using unconverted data
            return io.BytesIO(outfile.read())

    def upload_document_photo_background_video(self, document_photo, video_stream, mime_type):
        now = timezone.now()
        video = UserDocumentBackgroundVideo.objects.filter(photo=document_photo).first()

        with transaction.atomic(savepoint=False):
            if video is not None:

                # iOS hack.
                # Looks like iOS client send background video to old photo on resubmit.
                # Replace the requested photo with the most recent one in that case.
                if document_photo.verified_at is not None:
                    document_photo = (
                        UserDocumentPhoto.objects
                        .select_related('background_video')
                        .filter(
                            document_id=document_photo.document_id,
                            type=document_photo.type,
                        )
                        .order_by('-submitted_at')
                        .first()
                    )

                # Sanity check to reject late videos
                # but allow re-uploads until the video is verified.
                photo_has_video = document_photo.get_background_video() is not None
                photo_verified = document_photo.verified_at is not None
                if photo_has_video and photo_verified:
                    LOGGER.error(
                        'trying to upload background video to already verified photo %s',
                        document_photo.id,
                    )
                    return

                if video.mime_type != mime_type:
                    video.mime_type = mime_type
                    video.save()

            else:
                video = (
                    UserDocumentBackgroundVideo.objects
                    .create(
                        photo=document_photo,
                        mime_type=mime_type,
                        submitted_at=now,
                    )
                )
            app_install = self._user.app_installs.filter(is_latest=True).first()
            video_needs_conversion = (app_install is None
                                      or app_install.get_platform() is AppPlatform.ANDROID)
            if video_needs_conversion:
                video_stream = self.convert_x264(
                    video_stream, document_photo.id)
            self._mds_client.put_user_document_background_video(
                user_document_background_video=video,
                stream=video_stream,
            )

        self._log(
            subtype='background_video',
            data={
                'user_id': str(self._user.id),
                'background_video': {
                    'id': str(video.id),
                    'mime_type': mime_type,
                },
            },
        )

    def convert_document_photo_background_video(self, document_photo):
        video = document_photo.get_background_video()
        if video is None:
            LOGGER.error('not found video for photo_id %s', document_photo.id)
            return
        video_stream = self._mds_client.get_user_document_background_video(
            user_document_background_video=video,
        )['Body']
        converted_video_stream = self.convert_x264(video_stream, document_photo.id)
        self._mds_client.put_user_document_background_video(
            user_document_background_video=video,
            stream=converted_video_stream,
        )

    def update_passport_textdata(self, passport_data):
        old_passport_data = self._datasync_client.get_passport_unverified(
            self._user.uid, self._user.passport_ds_revision
        ) or {}
        passport_data['doc_type'] = 'id'

        # Handle the format in which the textdata is given out from API and the format it's sent
        for key in passport_data['registration']:
            if passport_data['registration'][key] is not None:
                passport_data[key] = passport_data['registration'][key]
        del passport_data['registration']

        passport_data = self._prepare_for_datasync_submission(
            old_data=old_passport_data,
            data=passport_data,
        )

        self._datasync_client.update_passport(self._user.uid, passport_data)

    def update_driving_license_textdata(self, driving_license_data):
        old_driving_license_data = self._datasync_client.get_license_unverified(
            self._user.uid,
            self._user.driving_license_ds_revision,
        ) or {}

        driving_license_data = self._prepare_for_datasync_submission(
            old_data=old_driving_license_data,
            data=driving_license_data,
        )

        self._datasync_client.update_license(self._user.uid, driving_license_data)

    def mark_plus_landing_viewed(self, performer_id=None):
        self._user.is_plus_screen_viewed = True
        self._user.updated_at = time.time()

        with transaction.atomic(savepoint=False):
            self._user.save()
            self._history_manager.update_entry(self._user, performer_id)

    def _prepare_for_datasync_submission(self, old_data, data):
        # Convert datetimes to isoformat and remove Nones
        for key in list(data):
            if isinstance(data[key], datetime.datetime):
                data[key] = data[key].isoformat()
            elif data[key] is None:
                del data[key]

        # Add missing fields
        for key, value in old_data.items():
            if key not in data:
                data[key] = value

        return data

    def _update_document_photo_content(self, document_type, photo_type, content):

        # Attach content to the latest doc if it's unverified or create a new one.
        document = (
            UserDocument.objects
            .filter(
                user=self._user,
                type=document_type.value,
            )
            .order_by('-submitted_at')
            .first()
        )
        if document is None or document.verified_at is not None:
            document = (
                UserDocument.objects
                .create(
                    user=self._user,
                    type=document_type.value,
                    submitted_at=timezone.now(),
                )
            )

        photo = (
            UserDocumentPhoto.objects
            .filter(
                document=document,
                type=photo_type.value,
            )
            .order_by('-submitted_at')
            .first()
        )
        if photo is None or photo.verified_at is not None:
            photo = (
                UserDocumentPhoto.objects
                .create(
                    document=document,
                    type=photo_type.value,
                    submitted_at=timezone.now(),
                    user=self._user,
                )
            )

        self._mds_client.put_user_document_photo(
            user_document_photo=photo,
            content=content,
        )

        self._log(
            subtype='document_photo',
            data={
                'user_id': str(self._user.id),
                'document_photo': {
                    'id': str(photo.id),
                    'type': photo_type.name,
                },
            },
        )

        return photo

    def _update_user_profile_from_driver_license_barcode(self, b64_barcode):
        try:
            b64_barcode = b64_barcode.encode('utf-8')
        except Exception:
            LOGGER.exception('failed to encode b64 barcode: %s', repr(b64_barcode))
            return

        try:
            binary_base64_barcode = base64.b64decode(b64_barcode)
        except ValueError:
            # Barcodes may come already decoded.
            LOGGER.info('failed to decode driver license barcode: %s', repr(b64_barcode))
            binary_base64_barcode = b64_barcode

        try:
            barcode = UserDrivingLicensePDF417Converter().convert_from_binary_data(
                binary_base64_barcode
            )
        except Exception:
            LOGGER.exception(
                'failed to parse driver license barcode: %s',
                repr(binary_base64_barcode),
            )
            return

        if barcode:
            if not self._user.first_name and 'first_name' in barcode:
                self.update_first_name(barcode['first_name'].capitalize(), commit=False)
            if not self._user.last_name and 'last_name' in barcode:
                self.update_last_name(barcode['last_name'].capitalize(), commit=False)
            if not self._user.patronymic_name and 'middle_name' in barcode:
                self.update_patronymic_name(barcode['middle_name'].capitalize(), commit=False)

            # commit has not performed yet
            self._user.updated_at = time.time()
            self._user.save()

            self._log(
                subtype='driver_license_barcode',
                data={
                    'user_id': str(self._user.id),
                },
            )

    def update_credit_card(self, pan_prefix=None, pan_suffix=None, paymethod_id=None):
        """
        Replace the preferred user credit card.
        The current card is logged to the audit table.
        The method will attempt to get paymethod_id from Trust.
        Raises CreditCardNotBound if the provided paymethod_id is not available in Trust.
        """

        assert (pan_prefix and pan_suffix) or paymethod_id

        if self._trust_client:
            payment_method = self._get_bound_payment_method(
                pan_prefix=pan_prefix,
                pan_suffix=pan_suffix,
                paymethod_id=paymethod_id,
            )
            if payment_method is None:
                raise self.CreditCardNotBound

            parts = payment_method['account'].split('*')
            pan_prefix = parts[0]
            pan_suffix = parts[-1]
            paymethod_id = payment_method['id']

        with transaction.atomic(savepoint=False):
            old_credit_card = UserCreditCard.objects.filter(user=self._user).first()
            if old_credit_card:
                credit_card_history = UserCreditCardHistory.from_credit_card(old_credit_card)
                old_credit_card.delete()
                credit_card_history.save()

            credit_card = UserCreditCard(
                user=self._user,
                paymethod_id=paymethod_id,
                pan_prefix=pan_prefix,
                pan_suffix=pan_suffix,
                bound_at=timezone.now(),
            )

        try:
            self._new_billing_client.link_card(user_id=self._user.id, paymethod_id=paymethod_id)
        except Exception as exc:
            LOGGER.exception(
                'error linking user credit card: user id - {}, credit card id - {}'
                .format(self._user.id, credit_card.id)
            )

#        self._user.credit_card = credit_card

        self._log(
            subtype='credit_card',
            data={
                'user_id': str(self._user.id),
                'credit_card': {
                    'id': str(credit_card.id),
                    'pan_prefix': credit_card.pan_prefix,
                    'pan_suffix': credit_card.pan_suffix,
                    'paymethod_id': credit_card.paymethod_id,
                },
            },
        )

        return credit_card

    def update_document_hashes(self, performer_id=None):
        passport = self._datasync_client.get_passport_unverified(self._user.uid, self._user.passport_ds_revision)
        license = self._datasync_client.get_license_unverified(self._user.uid, self._user.driving_license_ds_revision)

        is_modified = False
        if passport and 'doc_value' in passport and passport['doc_value']:
            new_hash = self._calculate_hash(passport['doc_value'].lower().replace(' ', ''))
        else:
            new_hash = None
        is_modified = is_modified or (new_hash != self._user.passport_number_hash)
        self._user.passport_number_hash = new_hash

        if license and 'number' in license and license['number']:
            new_hash = self._calculate_hash(license['number'].lower().replace(' ', ''))
        else:
            new_hash = None
        is_modified = is_modified or (new_hash != self._user.driving_license_number_hash)
        self._user.driving_license_number_hash = new_hash

        if is_modified:
            with transaction.atomic(savepoint=False):
                self._user.save()
                self._history_manager.update_entry(self._user, performer_id)

    def _calculate_hash(self, data):
        if not isinstance(data, bytes):
            data = data.encode('utf-8')
        out = hmac.new(cars.settings.DOCS_HASHES_KEY, data, sha1)
        result_bytes = out.digest()
        return binascii.hexlify(result_bytes).decode('utf-8')

    def add_tags(self, tags, performer_id=None):
        if self._user.tags is None:
            self._user.tags = []

        tags_before = set(self._user.tags)
        self._user.tags = list(set(self._user.tags) | set(tags))
        if tags_before == set(self._user.tags):
            return

        self._user.updated_at = time.time()

        with transaction.atomic(savepoint=False):
            self._user.save(update_fields=['tags', 'updated_at'])
            self._history_manager.update_entry(self._user, performer_id)

        self._log(
            subtype='tags.add',
            data={
                'tags': tags,
            },
        )

    def remove_tags(self, tags, performer_id=None):
        if self._user.tags is None:
            self._user.tags = []

        tags_before = set(self._user.tags)
        self._user.tags = list(set(self._user.tags) - set(tags))
        if tags_before == set(self._user.tags):
            return

        self._user.updated_at = time.time()

        with transaction.atomic(savepoint=False):
            self._user.save(update_fields=['tags', 'updated_at'])
            self._history_manager.update_entry(self._user, performer_id)

        self._log(
            subtype='tags.remove',
            data={
                'tags': tags,
            },
        )

    def update_plus_status(self, plus_status, performer_id=None):
        self._user.is_plus_user = plus_status
        self._user.updated_at = time.time()

        with transaction.atomic(savepoint=False):
            self._user.save()
            self._history_manager.update_entry(self._user, performer_id)

    def update_phone(self, phone, performer_id=None, use_proxy=False):
        if use_proxy:
            try:
                self._saas_client.update_phone(self._user.id, phone)
            except Exception:
                self._update_phone_internal(performer_id, phone)
        else:
            self._update_phone_internal(performer_id, phone)

    def _update_phone_internal(self, performer_id, phone):
        self._user.phone = phone
        self._user.updated_at = time.time()
        self._user.is_phone_verified = True

        self._user.save()
        self._history_manager.update_entry(self._user, performer_id)

    def request_phone_verification(self, performer_id=None):
        self._user.is_phone_verified = False
        self._user.updated_at = time.time()

        self._user.save()
        self._history_manager.update_entry(self._user, performer_id)

    def _get_bound_payment_method(self, pan_prefix=None, pan_suffix=None, paymethod_id=None):
        assert self._trust_client
        assert (pan_prefix and pan_suffix) or paymethod_id

        payment_methods = (
            self._trust_client.get_payment_methods(uid=self._user.uid)['bound_payment_methods']
        )

        bound_payment_method = None
        for payment_method in payment_methods:
            account = payment_method.get('account')
            if not account:
                LOGGER.warning('payment method with no account: %s', payment_method)
                continue

            if pan_prefix and pan_suffix:
                is_prefix_ok = account.startswith(pan_prefix)
                is_suffix_ok = account.endswith(pan_suffix)
            else:
                is_prefix_ok = is_suffix_ok = True

            if paymethod_id is None:
                is_paymethod_id_ok = True
            else:
                is_paymethod_id_ok = paymethod_id == payment_method['id']

            if is_prefix_ok and is_suffix_ok and is_paymethod_id_ok:
                bound_payment_method = payment_method
                break

        if bound_payment_method is None:
            LOGGER.warning(
                'user %s credit card payment method not found, available options are %s',
                self._user.id,
                payment_methods,
            )

        return bound_payment_method

    def _log(self, subtype, data):
        if self._push_client is None:
            return
        type_ = 'user_profile_updater.{}'.format(subtype)
        self._push_client.log(type_=type_, data=data)

    def _can_change_status(self, old_status, new_status):
        if new_status is User.Status.BLOCKED:
            can_change = old_status in {
                User.Status.ACTIVE,
                User.Status.DEBT,
                User.Status.ONBOARDING,
                User.Status.SCREENING,
            }
        elif old_status is User.Status.DEBT:
            can_change = new_status in {User.Status.ACTIVE, User.Status.BLOCKED}
        elif new_status is User.Status.DEBT:
            can_change = old_status is User.Status.ACTIVE
        elif old_status is User.Status.BLOCKED:
            can_change = new_status is User.Status.ACTIVE
        elif new_status is User.Status.REJECTED:
            can_change = old_status in {User.Status.ONBOARDING, User.Status.SCREENING}
        elif new_status is User.Status.ACTIVE and not cars.settings.IS_TESTS:
            is_unwanted_person_pass = False
            is_unwanted_person_license = False
            try:
                self.update_document_hashes()

                is_unwanted_person_pass = self._user.passport_number_hash is not None and ((
                    User.objects
                    .filter(
                        passport_number_hash=self._user.passport_number_hash,
                        status__in=[
                            User.Status.BLOCKED.value,
                            User.Status.DEBT.value,
                        ]
                    )
                    .exclude(id=self._user.id)
                    .count()
                ) > 0)

                is_unwanted_person_license = self._user.driving_license_number_hash is not None and ((
                    User.objects
                    .filter(
                        driving_license_number_hash=self._user.driving_license_number_hash,
                        status__in=[
                            User.Status.BLOCKED.value,
                            User.Status.DEBT.value,
                        ]
                    )
                    .exclude(id=self._user.id)
                    .count()
                ) > 0)
            finally:
                if is_unwanted_person_license or is_unwanted_person_pass:
                    twins = self._get_twins(
                        self._user.passport_number_hash,
                        self._user.driving_license_number_hash
                    )
                    all_tags = []
                    for twin in twins:
                        if twin.tags:
                            all_tags += twin.tags

                    blacklist_tags = [t for t in all_tags if t not in cars.settings.USERS['neutral_tags']]
                    can_change = len(blacklist_tags) == 0
                    LOGGER.info('there is unwanted user, who is similar to %s', str(self._user.id))
                else:
                    can_change = old_status is not new_status
        else:
            can_change = old_status is not new_status

        return can_change

    def _get_twins(self, passport_number_hash, driving_license_number_hash):
        qs = (
            User.objects
            .filter(
                Q(passport_number_hash=passport_number_hash) |
                Q(driving_license_number_hash=driving_license_number_hash)
            )
        )

        return qs


class RecognizedDocument(object):

    def __init__(self, scanner, type_, fields):
        self.scanner = scanner
        self.type = type_
        self.fields = fields


class RecognizedDocumentField(object):

    def __init__(self, value, confidence):
        self.value = value
        self.confidence = confidence
