import copy
import logging
import os
from enum import Enum

import kubiki

import cars.settings
from cars.core.daemons import CarsharingDaemon
from cars.core.util import make_yt_client
from cars.registration.models import RegistrationChatActionResult
from ..models import User
from ..serializers import AppInstallSerializer


LOGGER = logging.getLogger(__name__)


class RegistrationChatStatus(Enum):
    NOT_STARTED = 'not_started'
    IN_PROGRESS = 'in_progress'
    COMPLETED = 'completed'


class UserRegistrationChatData:

    def __init__(self, status, action_id):
        self.status = status
        self.action_id = action_id


class YtExportDaemon(CarsharingDaemon):

    tick_interval = '* * * * *'  # as soon as prev process ends

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._yt = make_yt_client('data')

    def get_distributed_lock_relative_path(self):
        return 'users/locks/users_yt_export.lock'

    def get_solomon_sensor_prefix(self):
        return 'users.yt_export'

    def get_solomon_service(self):
        return 'users'

    def get_distributed_lock(self):
        locks_config = copy.deepcopy(cars.settings.YT['locks'])
        base_dir = locks_config.pop('base_dir')
        locks_config['path'] = os.path.join(base_dir, self.get_distributed_lock_relative_path())

        locks_config['ttl'] = 500

        return kubiki.distributed_lock.YtLock(**locks_config)

    def _load_related(self, items, serializer_class, order_by):
        result = []
        for item in items.all():
            result.append(
                serializer_class(item).data
            )
        result.sort(key=lambda item: item[order_by])
        return result

    def _get_user_registration_chat_statuses(self):
        recent_chat_actions = (
            RegistrationChatActionResult.objects
            .only(
                'completed_at',
                'chat_action_id',
                'user__id',
            )
            .order_by(
                'user__id',
                '-submitted_at'
            )
            .distinct('user__id')
            .select_related('user')
        )

        users_registration_chat_status = {}
        for chat_action in recent_chat_actions:
            user_id = str(chat_action.user.id)
            if chat_action.completed_at is None:  # chat action in progress
                registration_chat_status = RegistrationChatStatus.IN_PROGRESS.value
            else:
                registration_chat_status = RegistrationChatStatus.COMPLETED.value
            users_registration_chat_status[user_id] = UserRegistrationChatData(
                status=registration_chat_status,
                action_id=chat_action.chat_action_id,
            )

        return users_registration_chat_status

    def _users_table_rows(self, users_registration_chat_status):
        all_users = (
            User.objects
            .only(
                'id',
                'uid',
                'status',
                'registration_geo',
            )
            .prefetch_related('app_installs')
        )

        for user in all_users:
            user_id = str(user.id)
            user_uid = user.uid
            user_status = user.status
            user_reg_chat_data = users_registration_chat_status.get(
                user_id,
                UserRegistrationChatData(
                    status=RegistrationChatStatus.NOT_STARTED.value,
                    action_id=None,
                ),
            )

            app_installs = self._load_related(
                items=user.app_installs,
                serializer_class=AppInstallSerializer,
                order_by='created_at',
            )

            if user.registration_geo is None:
                user_registration_geo = 'unknown'
            else:
                user_registration_geo = user.registration_geo

            user_data = {
                'id': user_id,
                'uid': user_uid,
                'status': user_status,
                'geo.registration': user_registration_geo,
                'registration_chat.status': user_reg_chat_data.status,
                'registration_chat.action_id': user_reg_chat_data.action_id,
                'app_installs': app_installs,
            }

            yield user_data

    def _do_tick(self):
        users_registration_chat_statuses = self._get_user_registration_chat_statuses()

        with self._yt.Transaction(timeout=50 * 60 * 1000, ping_ancestor_transactions=True):
            self._yt.write_table(
                cars.settings.USERS['export_table'],
                self._users_table_rows(users_registration_chat_statuses),
            )
