import itertools
import uuid  # pylint: disable=unused-import
import logging

from django.db import models

from cars.core.constants import AppPlatform
from .user import User


LOGGER = logging.getLogger(__name__)


class AppInstall(models.Model):

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    user = models.ForeignKey(
        User,
        db_index=False,
        on_delete=models.CASCADE,
        related_name='app_installs',
    )

    uuid = models.UUIDField()
    device_id = models.UUIDField()

    platform = models.CharField(max_length=1, choices=[(x.value, x.name) for x in AppPlatform])
    app_name = models.CharField(max_length=32)
    app_version = models.CharField(max_length=16, null=True)
    app_build = models.CharField(max_length=16, null=True)
    push_token = models.TextField(null=True, blank=True)

    is_latest = models.BooleanField()

    created_at = models.DateTimeField()
    refreshed_at = models.DateTimeField()

    class Meta:
        db_table = 'app_install'
        indexes = [
            models.Index(
                fields=['user_id'],
                name='app_install_user_id_idx',
            ),
            models.Index(
                fields=['uuid'],
                name='app_install_uuid_idx',
            ),
            models.Index(
                fields=['device_id'],
                name='app_install_device_id_idx',
            ),
        ]

    def __str__(self):
        return str(self.uuid)

    def get_platform(self):
        return AppPlatform(self.platform)

    def app_version_less_than(self, app_version):
        if self.app_version is None:
            return True

        self_parts = self.app_version.split('.')
        other_parts = app_version.split('.')

        for self_part, other_part in itertools.zip_longest(self_parts, other_parts):
            self_part = self_part or '0'
            other_part = other_part or '0'

            try:
                # Try cast to integers.
                self_part = int(self_part)
                other_part = int(other_part)
            except ValueError:
                # Cast back to strings.
                self_part = str(self_part)
                other_part = str(other_part)

            if self_part < other_part:
                return True
            if self_part > other_part:
                return False

        return False

    def app_build_less_than(self, app_build):
        if self.app_build is None:
            return True

        try:
            self_build = int(self.app_build)
            other_build = int(app_build)
        except ValueError:
            LOGGER.exception('unexpected app build numbers: %s and %s', self.app_build, app_build)
            return True

        return self_build < other_build
