import enum
import time
import uuid

from django.contrib.auth.models import AbstractBaseUser
from django.contrib.postgres.fields import ArrayField
from django.contrib.postgres.indexes import GinIndex
from django.core.exceptions import ObjectDoesNotExist
from django.db import models
from phonenumber_field.modelfields import PhoneNumberField

from cars.users.models.persdata_processing_consent import UserPersdataProcessingConsent
from cars.users.models.user_intro_screens import UserIntroScreens


class UserManager(models.Manager):

    def with_related(self):
        return (
            self.get_queryset()
        )


class User(AbstractBaseUser):

    class Status(enum.Enum):
        ACTIVE = 'active'
        BAD_AGE = 'bad_age'
        BAD_DRIVING_EXPERIENCE = 'bad_driving_exp'
        BLOCKED = 'blocked'
        DEBT = 'debt'
        DELETED = 'deleted'
        EXPIRED_DRIVER_LICENSE = 'expired_license'
        ONBOARDING = 'onboarding'
        REGISTRATION_FRAUD = 'reg_fraud'
        REJECTED = 'rejected'
        SCREENING = 'screening'

    USERNAME_FIELD = 'uid'

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)
    uid = models.BigIntegerField(unique=True)
    username = models.CharField(max_length=64, blank=True)
    email = models.EmailField(null=True)
    first_name = models.CharField(max_length=30, blank=True)
    last_name = models.CharField(max_length=150, blank=True)
    patronymic_name = models.CharField(max_length=64, blank=True)
    phone = PhoneNumberField(null=True)
    is_phone_verified = models.NullBooleanField(default=False)
    is_email_verified = models.NullBooleanField()
    status = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in Status],
        default=Status.ONBOARDING.value,
    )
    registered_at = models.DateTimeField(null=True, blank=True)
    date_joined = models.DateTimeField(null=True, blank=True)

    passport_number_hash = models.CharField(max_length=64, null=True)
    driving_license_number_hash = models.CharField(max_length=64, null=True)

    passport_ds_revision = models.CharField(max_length=64, null=True)
    driving_license_ds_revision = models.CharField(max_length=64, null=True)

    registration_geo = models.CharField(max_length=16, null=True)
    has_at_mark = models.NullBooleanField()

    is_first_riding = models.NullBooleanField(default=True)

    password = None
    last_login = None

    objects = UserManager()

    class Meta:
        db_table = 'user'
        indexes = [
            models.Index(
                fields=['email'],
                name='user_email_idx',
            ),
            models.Index(
                fields=['phone'],
                name='user_phone_idx',
            ),
            models.Index(
                fields=['registered_at'],
                name='user_registered_at_idx',
            ),
            models.Index(
                fields=['status'],
                name='user_status_idx',
            )
        ]

    def __str__(self):
        return self.username

    def has_tag(self, tag):
        return self.tags is not None and tag in self.tags  # pylint: disable=unsupported-membership-test

    def get_bonus_account(self):
        try:
            bonus_account = self.bonus_account
        except ObjectDoesNotExist:
            bonus_account = None
        return bonus_account

    def get_credit_card(self):
        try:
            credit_card = self.credit_card
        except ObjectDoesNotExist:
            credit_card = None
        return credit_card

    def get_registration_state(self):
        try:
            registration_state = self.registration_state
        except ObjectDoesNotExist:
            registration_state = None
        return registration_state

    def get_status(self):
        return self.Status(self.status)

    def get_new_backend_migration_shard(self):
        return int(str(self.id)[:4], 16)

    def get_print_name(self):
        print_name_parts = [
            field.capitalize().strip()
            for field in (self.first_name, self.last_name)
            if field
        ]
        return ' '.join(print_name_parts) if print_name_parts else self.username

    def get_full_name(self):
        full_name_parts = [
            field.capitalize().strip()
            for field in (self.last_name, self.first_name, self.patronymic_name)
            if field
        ]
        return ' '.join(full_name_parts) if full_name_parts else self.username

    def has_perm(self, *args, **kwargs):
        return True
