import uuid

from django.conf import settings
from django.db import models


class UserCreditCard(models.Model):
    id = models.UUIDField(default=uuid.uuid4, primary_key=True)
    user = models.OneToOneField(
        settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='credit_card',
    )
    paymethod_id = models.CharField(max_length=32, null=True)
    pan_prefix = models.CharField(max_length=6)
    pan_suffix = models.CharField(max_length=4)
    bound_at = models.DateTimeField()

    class Meta:
        db_table = 'user_credit_card'

    def __repr__(self):
        return '<UserCreditCard: pan={}, paymethod_id={}>'.format(
            self.get_masked_pan(),
            self.paymethod_id,
        )

    def get_masked_pan(self):
        return '{}-{}**-****-{}'.format(self.pan_prefix[:4], self.pan_prefix[4:], self.pan_suffix)


class UserCreditCardHistory(models.Model):
    id = models.UUIDField(default=uuid.uuid4, primary_key=True)
    user = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='credit_card_history',
    )
    paymethod_id = models.CharField(max_length=32, null=True)
    pan_prefix = models.CharField(max_length=6)
    pan_suffix = models.CharField(max_length=4)
    bound_at = models.DateTimeField()

    class Meta:
        db_table = 'user_credit_card_history'

    @classmethod
    def from_credit_card(cls, credit_card):
        return cls(
            id=credit_card.id,
            user=credit_card.user,
            paymethod_id=credit_card.paymethod_id,
            pan_prefix=credit_card.pan_prefix,
            pan_suffix=credit_card.pan_suffix,
            bound_at=credit_card.bound_at,
        )
