import enum
import hashlib
import uuid

from django.conf import settings
from django.contrib.postgres.fields import JSONField
from django.db import models

from cars.users.models import User


class UserDocument(models.Model):

    class Type(enum.Enum):
        DRIVER_LICENSE = 'dl'
        PASSPORT = 'p'

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)
    natural_id_fingerprint = models.CharField(max_length=64, null=True, blank=True)

    type = models.CharField(max_length=2, choices=[(x.value, x.name) for x in Type])

    submitted_at = models.DateTimeField()
    verified_at = models.DateTimeField(null=True, blank=True)

    datasync_id = models.CharField(max_length=32, unique=True, null=True, blank=True)
    unverified_datasync_id = models.CharField(max_length=32, unique=True, null=True, blank=True)

    user = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        on_delete=models.CASCADE,
        related_name='documents',
    )

    class Meta:
        db_table = 'user_document'
        indexes = [
            models.Index(
                fields=['natural_id_fingerprint'],
                name='user_document_natural_id_fingerprint_idx',
            ),
        ]

    def __str__(self):
        return self.Type(self.type).name

    @classmethod
    def get_natural_id_fingerprint(cls, natural_id):
        # First 64 of 128 bytes of hex-encoded sha512.
        return hashlib.sha512(natural_id.encode('utf-8')).hexdigest()[:64]


class UserDocumentPhoto(models.Model):

    class Type(enum.Enum):
        DRIVER_LICENSE_FRONT = 'lf'
        DRIVER_LICENSE_BACK = 'lb'
        DRIVER_LICENSE_SELFIE = 'ls'
        FACE = 'f'
        PASSPORT_BIOGRAPHICAL = 'pb'
        PASSPORT_REGISTRATION = 'pr'
        PASSPORT_SELFIE = 'ps'
        SELFIE = 'ss'

    class VerificationStatus(enum.Enum):
        UNRECOGNIZABLE = 'u'
        FOREIGN = 'f'
        OK = 'o'
        NEED_INFO = 'i'
        DISCARDED = 'd'
        NON_LATIN = 'n'
        VIDEO_SCREENCAP = 'v'
        VIDEO_ANOTHER_PERSON = 'a'
        VIDEO_ERROR = 'e'

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)
    type = models.CharField(max_length=2, choices=[(x.value, x.name) for x in Type])

    submitted_at = models.DateTimeField()
    verified_at = models.DateTimeField(null=True, blank=True)
    verification_status = models.CharField(
        max_length=1,
        choices=[(x.value, x.name) for x in VerificationStatus],
        null=True,
    )

    document = models.ForeignKey(
        UserDocument,
        on_delete=models.CASCADE,
        related_name='photos',
        null=True,
    )

    user = models.ForeignKey(
        User,
        on_delete=models.CASCADE,
        related_name='photos',
        null=True,
    )

    class Meta:
        db_table = 'user_document_photo'
        indexes = [
            models.Index(
                fields=['verified_at'],
                name='user_document_photo_verified_at_idx',
            ),
        ]

    def __str__(self):
        return self.Type(self.type).name

    def get_background_video(self):
        try:
            background_video = self.background_video
        except UserDocumentBackgroundVideo.DoesNotExist:
            background_video = None
        return background_video

    def get_type(self):
        return self.Type(self.type)

    def get_verification_status(self):
        return self.VerificationStatus(self.verification_status)


class UserDocumentBackgroundVideo(models.Model):
    """Background video captured before taking a photo"""

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)
    photo = models.OneToOneField(
        UserDocumentPhoto,
        on_delete=models.CASCADE,
        related_name='background_video',
    )
    mime_type = models.CharField(max_length=16)
    submitted_at = models.DateTimeField()

    class Meta:
        db_table = 'user_document_background_video'
