#!/usr/bin/env python
import argparse
import logging.config
import os

import cars.periodic
import cars.settings


def main():
    logging.config.dictConfig(cars.settings.LOGGING)

    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers()

    parser_daily_operator_stats = subparsers.add_parser('daily_operator_stats')
    parser_daily_operator_stats.set_defaults(func=run_daily_operator_stats)

    parser_build_orders = subparsers.add_parser('build_orders')
    parser_build_orders.set_defaults(func=run_build_orders)

    parser_estimate_car_pool = subparsers.add_parser('estimate_car_pool')
    parser_estimate_car_pool.set_defaults(func=run_estimate_car_pool)

    parser_radar = subparsers.add_parser('radar')
    parser_radar.set_defaults(func=run_radar)

    parser_telematics_server = subparsers.add_parser('telematics_server')
    parser_telematics_server.add_argument('--backend_port', type=int, required=True)
    parser_telematics_server.add_argument('--frontend_port', type=int, required=True)
    parser_telematics_server.set_defaults(func=run_telematics_server)

    test_registration = subparsers.add_parser('test_registration')
    test_registration.add_argument('--script', required=True)
    test_registration.set_defaults(func=run_test_registration)

    args = parser.parse_args()
    args.func(args)


def run_build_orders(_):
    cars.periodic.build_orders.main()

def run_daily_operator_stats(_):
    cars.periodic.daily_operator_stats.main()

def run_estimate_car_pool(_):
    cars.periodic.estimate_car_pool.main()


def run_radar(_):
    import django
    os.environ['DJANGO_SETTINGS_MODULE'] = 'cars.django.settings'
    django.setup()

    from cars.radar.daemon import RadarDaemon

    saas_config = cars.settings.CAR_INDEX['SAAS_CONFIG']
    radar = RadarDaemon(saas_config=saas_config,
                        remind_interval=cars.settings.RADAR['remind_interval'])
    radar.run()


def run_telematics_server(args):
    # Lazy load telematics module to avoid problems with Twisted on YT.
    import cars.telematics  # pylint: disable=redefined-outer-name
    cars.telematics.server.main(
        backend_port=args.backend_port,
        frontend_port=args.frontend_port,
    )


def run_test_registration(args):
    from cars.registration.core.chat.script import RegistrationScript
    with open(args.script) as f:
        script = RegistrationScript.from_yaml(f)
    print(script)


if __name__ == '__main__':
    main()
