#!/usr/bin/env python
import os
import re
import subprocess
import tempfile

import requests

PROJECT = 'carsharing'

LANGUAGES = [
    'en',
    'hu',
    'ru',
]

KEYSETS = [
    'Backend.Orders',
    'Backend.Fines',
    'CarModel',
    'City',
    'Operator',
]

NEWLINE_MARKER = '__newline__'

LOCALE_DIR = 'cars/messages'

TANKER_URL_TEMPLATE = 'https://tanker.yandex-team.ru/~api/download?branch=master&format=po&include-unapproved=yes&languages={language}&keysets={keyset}&project={project}'  # pylint: disable=line-too-long


def main():
    for language in LANGUAGES:
        with tempfile.TemporaryDirectory() as lang_dir:
            process_language(language, lang_dir)


def process_language(language, workdir):
    files_to_merge = []
    for keyset in KEYSETS:
        keyset_path = '{}.po'.format(os.path.join(workdir, keyset))
        fetch_keyset(language, keyset, keyset_path)
        files_to_merge.append(keyset_path)

    for domain in ['default', 'django']:
        output_filename = '{}.po'.format(domain)
        output_file = os.path.join(LOCALE_DIR, language, 'LC_MESSAGES', output_filename)
        merge_keysets(files_to_merge, output_file)
        fix_newlines(output_file, output_file)


def fetch_keyset(language, keyset, output_path):
    url = TANKER_URL_TEMPLATE.format(project=PROJECT, language=language, keyset=keyset)
    r = requests.get(url, verify=False)
    r.raise_for_status()
    with open(output_path, 'w') as f:
        f.write(r.text)


def merge_keysets(input_files, output_file):
    output_dir = os.path.dirname(output_file)
    os.makedirs(output_dir, exist_ok=True)

    cmd = [
        'msgcat',
        '--output-file', output_file,
    ] + input_files
    retcode = subprocess.call(cmd)
    assert retcode == 0, 'Command failed: "{}"'.format(' '.join(cmd))


def fix_newlines(input_file, output_file):
    with open(input_file) as f:
        data = f.read()
    data = re.sub(NEWLINE_MARKER, r'\\n', data)
    with open(output_file, 'w') as f:
        f.write(data)


if __name__ == '__main__':
    main()
