import argparse
import logging
import os
import random
import string
import time
import urllib.error
import uuid

from drive.backend.api import client as api
from drive.qatool.tus.tus import TestUserService as TusClient
from drive.qatool.events import TestEvent as testEvent


def _generate_login():
    return 'yndx-drive-{0}.{1}'.format(_random_numeric(5), _random_numeric(5))


def _random_numeric(n_digits):
    return ''.join(random.choice(string.digits) for _ in range(n_digits))


def _create_tus_account(tus_client, login=None, password=None, firstname=None, lastname=None, language=None,
                        country=None, tags=None,
                        tus_consumer=None):
    if not login:
        login = _generate_login()
    if not password:
        password = "simple123456"
    account = tus_client.create_account(login, password, firstname, lastname, language, country, tags, tus_consumer)
    logging.info("created account {}".format(account.login))
    return account


def _add_account(client, username, status, firstname, lastname, uid, email):
    search_result = client.search_users(username)
    if search_result:
        logging.info("username account already exists")
        return None
    user_id = str(uuid.uuid4())
    user = client.edit_user(user_id=user_id, username=username, status=status,
                            first_name=firstname, last_name=lastname, uid=uid, email=email)

    logging.info("The account {} is succesfully added".format(username))
    return user


def _search_users(client, username):
    user = client.search_users(username)[0]
    if not user or user[0].username != username:
        logging.info("Cannot find account in system".format(username))
        return None
    return user


def _edit_account(client, username):
    user = _search_users(client, username)
    if user:
        user_id = user.id
        client.edit_user(user_id=user_id, username=username)
        logging.info("Account {} is succesfully cleared".format(username))


def _add_role(client, role_id, user_id, active):
    client.add_user_role(role_id=role_id, active=active, user_id=user_id)


def _add_bonuses(client, user_id, amount=0):
    client.add_user_tag(user_id=user_id, tag_name="test_bb_credit", amount=amount)


def _is_correct_role(client, role_id):
    roles = client.list_roles()
    for role in roles:
        if role["role_id"] == role_id:
            return True
    return False


def _save_accounts_to_file(users_list, dir_path):
    file_name = "accounts_" + time.strftime("%Y%m%d-%H%M%S") + ".txt"

    path = os.path.join(dir_path, file_name)

    fields = "Аккаунт;Пароль;Статус;Есть бонусы\n"
    if users_list:
        with open(path, "w") as file:
            file.write(fields)
            for user in users_list:
                file.write(f'{user.username};{user.password};{user.status};"Да"\n')

        logging.info(f'file with accounts saved in: {path}')
    else:
        logging.info(f'There is no accounts')


def _main_create_accounts(client, tus_client, status, roles_list, counter, dir_path):
    users_list = []
    for i in range(counter):
        try:
            tus_account = _create_tus_account(tus_client=tus_client)
            assert tus_account
            login = tus_account.login
            uid = int(tus_account.uid)
            email = login + "@yandex.ru"
            user = _add_account(client, login, status=status, firstname="Тестовый",
                                lastname="Тестович", uid=uid, email=email)
            user.password = tus_account.password
            if roles_list:
                for role in roles_list:
                    _add_role(client, role, user.id, True)
            users_list.append(user)
            _add_bonuses(client=client, user_id=user.id, amount=100000000)
        except urllib.error.HTTPError:
            continue
    for user in users_list:
        testEvent(user.username, user.password, user.status).report_json()
    if dir_path:
        _save_accounts_to_file(users_list, dir_path)


def main():
    parser = argparse.ArgumentParser(description="Drive Client")
    parser.add_argument('-y', "--public-token", dest="public_token", metavar="OAUTH", type=str, default=None,
                        help="Yandex.Passport OAuth token")
    parser.add_argument('-s', "--private-token", dest="private_token", metavar="OATH", type=str, default=None,
                        help="Internal Passport OAuth token")
    parser.add_argument('-v', "--verbose", dest="verbose", action="store_true", help="enable debug level logging")

    subparsers = parser.add_subparsers(help="modes")
    create_account_parser = subparsers.add_parser("create_accounts", help="create users accounts")
    create_account_parser.add_argument("--tus-token", dest="tus_token", metavar="tus_token,", type=str, required=True)
    create_account_parser.add_argument("-r", "--role", dest="roles_id", metavar="roles_id", type=str)
    create_account_parser.add_argument("--status", dest="status_name", metavar="status_name",
                                       choices=['active', 'onboarding'], type=str, required=True)
    create_account_parser.add_argument("-n", "--number", dest="number", metavar="number", type=int, required=True,
                                       help="number of accounts")
    create_account_parser.add_argument("-o", dest="output_dir", type=str, help="directory name")
    create_account_parser.add_argument("-s", "--source", dest="source", choices=['qa', 'testing'], type=str,
                                       required=True, help="source destination")

    args = parser.parse_args()

    logging_level = logging.DEBUG if args.verbose else logging.INFO
    logging.basicConfig(level=logging_level)

    sources = {"qa": "https://testing.carsharing.yandex.net?backend_cluster=qa",
               "testing": "https://testing.carsharing.yandex.net"}
    endpoint_source = sources.get(args.source) if args.source in sources.keys() else args.source
    endpoint = endpoint_source
    tus_token = args.tus_token

    client = api.BackendClient(endpoint=endpoint, public_token=args.public_token,
                               private_token=args.private_token)

    roles_list = [x.strip() for x in args.roles_id.split(',')] if args.roles_id else None
    status = args.status_name
    counter = args.number
    dir_path = args.output_dir if args.output_dir else None

    if roles_list:
        for role in roles_list:
            if _is_correct_role(client, role):
                continue
            else:
                logging.error(f'"{role}" is incorrect, please check that the role is available')
                raise NameError
    if dir_path:
        if not os.path.isdir(dir_path):
            logging.error(f'incorrect directory path: {dir_path}')
            raise OSError
    tus_client = TusClient(token=tus_token)
    _main_create_accounts(client=client, status=status,
                          roles_list=roles_list, counter=counter,
                          dir_path=dir_path, tus_client=tus_client)
