package api

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strconv"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
)

type ActionOld struct {
	models.Action
	// Tasks contains list of action tasks.
	Tasks []models.Task `json:""`
}

func (v *View) GetActionOld(c echo.Context) error {
	action, ok := c.Get(actionKey).(models.Action)
	if !ok {
		return fmt.Errorf("action not extracted")
	}
	tasks, err := v.core.Tasks.FindByAction(action.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if tasks == nil {
		tasks = make([]models.Task, 0)
	}
	return c.JSON(http.StatusOK, ActionOld{
		Action: action,
		Tasks:  tasks,
	})
}

type ActionForm struct {
	DirID       *int    `json:""`
	Title       string  `json:""`
	Description *string `json:""`
	// Options contains action options.
	Options *models.ActionOptions `json:""`
}

func (f *ActionForm) Update(action *models.Action) {
	if f.DirID != nil {
		action.DirID = models.NInt(*f.DirID)
	}
	if f.Title != "" {
		action.Title = f.Title
	}
	if f.Description != nil {
		action.Description = *f.Description
	}
	if f.Options != nil {
		action.Options = *f.Options
	}
}

func (v *View) CreateActionOld(c echo.Context) error {
	var form ActionForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.NoContent(http.StatusBadRequest)
	}
	var action models.Action
	form.Update(&action)
	if err := v.core.Actions.Validate(action); err != nil {
		if err, ok := err.(models.FieldListError); ok {
			c.Logger().Warn(err)
			return c.JSON(http.StatusBadRequest, ErrorResponse{
				FieldErrors: err,
			})
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	account, ok := c.Get(authAccountKey).(models.Account)
	if !ok {
		return fmt.Errorf("account not extracted")
	}
	action.OwnerID = models.NInt(account.ID)
	action.CreateTime = time.Now().Unix()
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Actions.CreateTx(
				tx, &action, getEventOptions(c)...,
			)
		},
	); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusCreated, action)
}

func (v *View) UpdateActionOld(c echo.Context) error {
	var form ActionForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.NoContent(http.StatusBadRequest)
	}
	action, ok := c.Get(actionKey).(models.Action)
	if !ok {
		return fmt.Errorf("action not extracted")
	}
	form.Update(&action)
	if err := v.core.Actions.Validate(action); err != nil {
		if err, ok := err.(models.FieldListError); ok {
			c.Logger().Warn(err)
			return c.JSON(http.StatusBadRequest, ErrorResponse{
				FieldErrors: err,
			})
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Actions.UpdateTx(
				tx, action, getEventOptions(c)...,
			)
		},
	); err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusOK, action)
}

func (v *View) RemoveActionOld(c echo.Context) error {
	action, ok := c.Get(actionKey).(models.Action)
	if !ok {
		return fmt.Errorf("action not extracted")
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Actions.RemoveTx(
				tx, action.ID, getEventOptions(c)...,
			)
		},
	); err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.NoContent(http.StatusOK)
}

func (v *View) ExecuteActionOld(c echo.Context) error {
	action, ok := c.Get(actionKey).(models.Action)
	if !ok {
		return fmt.Errorf("action not extracted")
	}
	requestBody, err := ioutil.ReadAll(c.Request().Body)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	var request struct {
		Options models.TaskOptions `json:""`
	}
	err = json.Unmarshal(requestBody, &request)
	if err != nil {
		c.Logger().Error(err)
		return echo.NewHTTPError(
			http.StatusBadRequest,
			fmt.Sprintf("Entered invalid request body: %s", err),
		)
	}
	account, ok := c.Get(authAccountKey).(models.Account)
	if !ok {
		return fmt.Errorf("account not extracted")
	}
	task := models.Task{
		ActionID: action.ID,
		OwnerID:  models.NInt(account.ID),
		Options:  request.Options,
	}
	if err := v.core.Tasks.Create(&task); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusOK, task)
}

type Action struct {
	ID          int                  `json:"id"`
	NodeID      int                  `json:"node_id,omitempty"`
	Title       string               `json:"title"`
	Description string               `json:"description"`
	Options     models.ActionOptions `json:"options"`
}

func (v *View) ObserveAction(c echo.Context) error {
	action, ok := c.Get(actionKey).(models.Action)
	if !ok {
		return fmt.Errorf("action not extracted")
	}
	return c.JSON(http.StatusOK, makeAction(action))
}

type UpdateActionForm struct {
	NodeID      *int                  `json:"node_id"`
	Title       *string               `json:"title"`
	Description *string               `json:"description"`
	Options     *models.ActionOptions `json:"options"`
}

func (f UpdateActionForm) Update(action *models.Action) *ErrorResponse {
	if f.NodeID != nil {
		action.DirID = models.NInt(*f.NodeID)
	}
	if f.Title != nil {
		action.Title = *f.Title
	}
	if f.Description != nil {
		action.Description = *f.Description
	}
	if f.Options != nil {
		action.Options = *f.Options
	}
	return nil
}

func makeAction(action models.Action) Action {
	return Action{
		ID:          action.ID,
		NodeID:      int(action.DirID),
		Title:       action.Title,
		Description: action.Description,
		Options:     action.Options,
	}
}

func (v *View) CreateAction(c echo.Context) error {
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		c.Logger().Error("permissions not extracted")
		return fmt.Errorf("permissions not extracted")
	}
	account, ok := c.Get(authAccountKey).(models.Account)
	if !ok {
		c.Logger().Error("account not extracted")
		return fmt.Errorf("account not extracted")
	}
	var form UpdateActionForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	action := models.Action{
		OwnerID: models.NInt(account.ID),
	}
	if err := form.Update(&action); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := v.core.Actions.Validate(action); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	if action.DirID != 0 {
		if parent, err := v.core.Nodes.Get(int(action.DirID)); err == nil {
			permissions = v.extendNodePermissions(c, permissions, parent)
		} else if err != sql.ErrNoRows {
			c.Logger().Error(err)
			return err
		}
	}
	if !permissions.HasPermission(models.CreateActionPermission) {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message:            "account missing permissions",
			MissingPermissions: []string{models.CreateActionPermission},
		})
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Actions.CreateTx(
				tx, &action, getEventOptions(c)...,
			)
		},
	); err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	return c.JSON(http.StatusCreated, makeAction(action))
}

func (v *View) UpdateAction(c echo.Context) error {
	action, ok := c.Get(actionKey).(models.Action)
	if !ok {
		return fmt.Errorf("action not extracted")
	}
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		return fmt.Errorf("permissions not extracted")
	}
	var form UpdateActionForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	if err := form.Update(&action); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := v.core.Actions.Validate(action); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	if action.DirID != 0 {
		node, err := v.core.Nodes.Get(int(action.DirID))
		if err != nil {
			return err
		}
		permissions = v.extendNodePermissions(c, permissions, node)
	}
	if !permissions.HasPermission(models.CreateActionPermission) {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message:            "account missing permissions",
			MissingPermissions: []string{models.CreateActionPermission},
		})
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Actions.UpdateTx(
				tx, action, getEventOptions(c)...,
			)
		},
	); err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	return c.JSON(http.StatusOK, makeAction(action))
}

type ObserveActionTasksForm struct {
	Begin int `query:"begin"`
	Limit int `query:"limit"`
}

func (v *View) ObserveActionTasks(c echo.Context) error {
	action, ok := c.Get(actionKey).(models.Action)
	if !ok {
		return fmt.Errorf("planner not extracted")
	}
	form := ObserveActionTasksForm{}
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf("Unable to parse request: %q", err.Error()),
		})
	}
	if form.Limit <= 0 {
		form.Limit = 100
	} else if form.Limit > 1000 {
		form.Limit = 1000
	}
	tasks, nextBegin, err := v.core.Tasks.FindPageByAction(
		action.ID, models.NInt(form.Begin), form.Limit,
	)
	if err != nil {
		return err
	}
	resp := TasksResponse{NextBegin: int64(nextBegin)}
	for _, task := range tasks {
		resp.Tasks = append(resp.Tasks, makeTask(task))
	}
	return c.JSON(http.StatusOK, resp)
}

type RunActionForm struct {
	Options models.TaskOptions `json:"options"`
}

func (v *View) RunAction(c echo.Context) error {
	action, ok := c.Get(actionKey).(models.Action)
	if !ok {
		return fmt.Errorf("action not extracted")
	}
	account, ok := c.Get(authAccountKey).(models.Account)
	if !ok {
		return fmt.Errorf("account not extracted")
	}
	var form RunActionForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	task := models.Task{
		ActionID: action.ID,
		OwnerID:  models.NInt(account.ID),
		Options:  form.Options,
	}
	if err := v.core.Tasks.Create(&task); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusCreated, makeTask(task))
}

type PlanActionForm struct {
	Title       string                 `json:"title"`
	Description string                 `json:"description"`
	Options     models.TaskOptions     `json:"options"`
	Settings    models.PlannerSettings `json:"settings"`
}

func (v *View) PlanAction(c echo.Context) error {
	action, ok := c.Get(actionKey).(models.Action)
	if !ok {
		return fmt.Errorf("action not extracted")
	}
	account, ok := c.Get(authAccountKey).(models.Account)
	if !ok {
		return fmt.Errorf("account not extracted")
	}
	var form PlanActionForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	planner := models.Planner{
		ActionID:    action.ID,
		OwnerID:     account.ID,
		DirID:       int(action.DirID),
		Title:       form.Title,
		Description: form.Description,
		Options:     form.Options,
		Settings:    form.Settings,
		CreateTime:  time.Now().Unix(),
	}
	if err := v.core.WithTx(c.Request().Context(), func(tx *sql.Tx) error {
		return v.core.Planners.CreateTx(tx, &planner, getEventOptions(c)...)
	}); err != nil {
		c.Logger().Error(err)
		return err
	}
	resp := makePlanner(planner)
	if state, err := v.core.PlannerStates.Get(planner.ID); err == nil {
		resp.NextTime = int64(state.NextTime)
	} else if err != sql.ErrNoRows {
		c.Logger().Error("Error:", err)
	}
	return c.JSON(http.StatusCreated, resp)
}

func (v *View) registerActions(g *echo.Group) {
	g.POST(
		"/v0/actions", v.CreateAction, v.sessionAuth, v.requireAuth,
		v.requirePermission(),
	)
	g.GET(
		"/v0/actions/:action", v.ObserveAction, v.sessionAuth, v.extractAction,
		v.requireNodePermission(models.ObserveActionPermission),
	)
	g.PATCH(
		"/v0/actions/:action", v.UpdateAction, v.sessionAuth, v.requireAuth, v.extractAction,
		v.requireNodePermission(models.UpdateActionPermission),
	)
	g.POST(
		"/v0/actions/:action/run", v.RunAction, v.sessionAuth, v.requireAuth, v.extractAction,
		v.requireNodePermission(models.RunActionPermission),
	)
	g.GET(
		"/v0/actions/:action/tasks", v.ObserveActionTasks, v.sessionAuth, v.extractAction,
		v.requireNodePermission(models.ObserveActionPermission, models.ObserveTaskPermission),
	)
	g.POST(
		"/v0/actions/:action/plan", v.PlanAction, v.sessionAuth, v.requireAuth, v.extractAction,
		v.requireNodePermission(models.CreatePlannerPermission),
	)
	// Deprecated.
	g.POST(
		"/actions", v.CreateActionOld, v.sessionAuth, v.requireAuth,
		v.requirePermission(models.LegacyAuthPermission),
	)
	g.GET(
		"/actions/:action", v.GetActionOld, v.sessionAuth, v.extractAction,
		v.requireNodePermission(models.ObserveActionPermission),
	)
	g.PATCH(
		"/actions/:action", v.UpdateActionOld, v.sessionAuth, v.requireAuth, v.extractAction,
		v.requireNodePermission(models.UpdateActionPermission),
	)
	g.DELETE(
		"/actions/:action", v.RemoveActionOld, v.sessionAuth, v.requireAuth, v.extractAction,
		v.requireNodePermission(models.DeleteActionPermission),
	)
	g.POST(
		"/actions/:action", v.ExecuteActionOld, v.sessionAuth, v.requireAuth, v.extractAction,
		v.requireNodePermission(models.RunActionPermission),
	)
}

func (v *View) extractAction(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		id, err := strconv.Atoi(c.Param("action"))
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusBadRequest)
		}
		action, err := v.core.Actions.Get(id)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		c.Set(actionKey, action)
		if action.DirID != 0 {
			node, err := v.core.Nodes.Get(int(action.DirID))
			if err != nil {
				if err == sql.ErrNoRows {
					return c.NoContent(http.StatusNotFound)
				}
				c.Logger().Error(err)
				return err
			}
			c.Set(nodeKey, node)
		}
		return next(c)
	}
}
