package api

import (
	"database/sql"
	"fmt"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
)

type Config struct {
	ID          int    `json:"id"`
	NodeID      int    `json:"node_id,omitempty"`
	Title       string `json:"title"`
	Description string `json:"description"`
	Data        string `json:"data"`
}

func (v *View) ObserveConfig(c echo.Context) error {
	config, ok := c.Get(configKey).(models.Config)
	if !ok {
		return fmt.Errorf("config not extracted")
	}
	return c.JSON(http.StatusOK, makeConfig(config))
}

type UpdateConfigForm struct {
	NodeID      *int    `json:"node_id"`
	Title       *string `json:"title"`
	Description *string `json:"description"`
	Data        *string `json:"data"`
}

func (f UpdateConfigForm) Update(config *models.Config) *ErrorResponse {
	if f.NodeID != nil {
		config.DirID = models.NInt(*f.NodeID)
	}
	if f.Title != nil {
		config.Title = *f.Title
	}
	if f.Description != nil {
		config.Description = *f.Description
	}
	if f.Data != nil {
		config.Data = *f.Data
	}
	return nil
}

func makeConfig(config models.Config) Config {
	return Config{
		ID:          config.ID,
		NodeID:      int(config.DirID),
		Title:       config.Title,
		Description: config.Description,
		Data:        config.Data,
	}
}

func (v *View) CreateConfig(c echo.Context) error {
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		c.Logger().Error("permissions not extracted")
		return fmt.Errorf("permissions not extracted")
	}
	var form UpdateConfigForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	config := models.Config{}
	if err := form.Update(&config); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := v.core.Configs.Validate(config); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	if config.DirID != 0 {
		if parent, err := v.core.Nodes.Get(int(config.DirID)); err == nil {
			permissions = v.extendNodePermissions(c, permissions, parent)
		} else if err != sql.ErrNoRows {
			c.Logger().Error(err)
			return err
		}
	}
	if !permissions.HasPermission(models.CreateConfigPermission) {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message:            "account missing permissions",
			MissingPermissions: []string{models.CreateConfigPermission},
		})
	}
	if err := v.core.Configs.CreateTx(
		v.core.DB, &config, getEventOptions(c)...,
	); err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	return c.JSON(http.StatusCreated, makeConfig(config))
}

func (v *View) UpdateConfig(c echo.Context) error {
	config, ok := c.Get(configKey).(models.Config)
	if !ok {
		return fmt.Errorf("config not extracted")
	}
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		return fmt.Errorf("permissions not extracted")
	}
	var form UpdateConfigForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	if err := form.Update(&config); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := v.core.Configs.Validate(config); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	if config.DirID != 0 {
		node, err := v.core.Nodes.Get(int(config.DirID))
		if err != nil {
			return err
		}
		permissions = v.extendNodePermissions(c, permissions, node)
	}
	if !permissions.HasPermission(models.CreateConfigPermission) {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message:            "account missing permissions",
			MissingPermissions: []string{models.CreateConfigPermission},
		})
	}
	if err := v.core.Configs.UpdateTx(
		v.core.DB, config, getEventOptions(c)...,
	); err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	return c.JSON(http.StatusOK, makeConfig(config))
}

func (v *View) DeleteConfig(c echo.Context) error {
	config, ok := c.Get(configKey).(models.Config)
	if !ok {
		return fmt.Errorf("config not extracted")
	}
	if err := v.core.Configs.RemoveTx(
		v.core.DB, config.ID, getEventOptions(c)...,
	); err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	return c.JSON(http.StatusOK, makeConfig(config))
}

func (v *View) registerConfigs(g *echo.Group) {
	g.GET(
		"/v0/configs/:config", v.ObserveConfig, v.sessionAuth, v.extractConfig,
		v.requireNodePermission(models.ObserveConfigPermission),
	)
	g.POST(
		"/v0/configs", v.CreateConfig, v.sessionAuth, v.requireAuth,
		v.requirePermission(),
	)
	g.PATCH(
		"/v0/configs/:config", v.UpdateConfig, v.sessionAuth, v.requireAuth, v.extractConfig,
		v.requireNodePermission(models.UpdateConfigPermission),
	)
	g.DELETE(
		"/v0/configs/:config", v.DeleteConfig, v.sessionAuth, v.requireAuth, v.extractConfig,
		v.requireNodePermission(models.DeleteConfigPermission),
	)
}

func (v *View) extractConfig(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		id, err := strconv.Atoi(c.Param("config"))
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusBadRequest)
		}
		config, err := v.core.Configs.Get(id)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		c.Set(configKey, config)
		if config.DirID != 0 {
			node, err := v.core.Nodes.Get(int(config.DirID))
			if err != nil {
				if err == sql.ErrNoRows {
					return c.NoContent(http.StatusNotFound)
				}
				c.Logger().Error(err)
				return err
			}
			c.Set(nodeKey, node)
		}
		return next(c)
	}
}
