package api

import (
	"net/http"
	"sort"

	"a.yandex-team.ru/drive/runner/models"
	"github.com/labstack/echo/v4"
)

// Host represents host.
type Host struct {
	ID       int               `json:"id"`
	Name     string            `json:"name"`
	PingTime int64             `json:"ping_time"`
	Config   models.HostConfig `json:"config"`
	State    models.HostState  `json:"state"`
	Running  bool              `json:"running"`
}

// GetHostResp represents GetHosts response.
type GetHostResp struct {
	// Hosts contains list of hosts.
	Hosts []Host `json:"hosts"`
}

// GetHosts returns list of hosts.
func (v *View) GetHosts(c echo.Context) error {
	hosts, err := v.core.Hosts.All()
	if err != nil {
		return err
	}
	var resp GetHostResp
	for _, host := range hosts {
		respHost := Host{
			ID:       host.ID,
			Name:     host.Name,
			PingTime: host.PingTime,
			Config:   host.Config,
			State:    host.State,
			Running:  true,
		}
		if !host.Running() {
			respHost.Running = false
			respHost.State.RunningTasks = nil
		}
		resp.Hosts = append(resp.Hosts, respHost)
	}
	sort.Sort(hostSorter(resp.Hosts))
	return c.JSON(http.StatusOK, resp)
}

func (v *View) registerHosts(g *echo.Group) {
	g.GET(
		"/v0/hosts", v.GetHosts, v.sessionAuth,
		v.requirePermission(models.ObserveHostPermission),
	)
}

type hostSorter []Host

func (a hostSorter) Len() int {
	return len(a)
}

func (a hostSorter) Swap(i, j int) {
	a[i], a[j] = a[j], a[i]
}

func (a hostSorter) Less(i, j int) bool {
	return a[i].Name < a[j].Name
}
