package api

import (
	"context"
	"database/sql"
	"fmt"
	"net/http"

	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
	"github.com/labstack/echo/v4"
)

type UpdateNodeForm struct {
	NodeID       *models.NInt `json:"node_id"`
	Name         *string      `json:"name"`
	Description  *string      `json:"description"`
	InheritRoles *bool        `json:"inherit_roles"`
}

func (f UpdateNodeForm) Update(node *models.Node) *ErrorResponse {
	invalidFields := InvalidFields{}
	if f.NodeID != nil {
		node.NodeID = *f.NodeID
	}
	if f.Name != nil {
		node.Title = *f.Name
	}
	if f.Description != nil {
		node.Description = *f.Description
	}
	if f.InheritRoles != nil {
		node.InheritRoles = *f.InheritRoles
	}
	if len(node.Title) < 3 {
		invalidFields["name"] = InvalidField{Message: "node name is too short"}
	}
	if len(node.Title) > 32 {
		invalidFields["name"] = InvalidField{Message: "node name is too long"}
	}
	if len(node.Description) > 1024 {
		invalidFields["description"] = InvalidField{Message: "node description is too long"}
	}
	if len(invalidFields) > 0 {
		return &ErrorResponse{
			Message:       "form has invalid fields",
			InvalidFields: invalidFields,
		}
	}
	return nil
}

type Node struct {
	ID           int    `json:"id"`
	NodeID       int    `json:"node_id,omitempty"`
	Name         string `json:"name"`
	Description  string `json:"description"`
	InheritRoles bool   `json:"inherit_roles"`
}

func makeNode(node models.Node) Node {
	return Node{
		ID:           node.ID,
		NodeID:       int(node.NodeID),
		Name:         node.Title,
		Description:  node.Description,
		InheritRoles: node.InheritRoles,
	}
}

func (v *View) CreateNode(c echo.Context) error {
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		return fmt.Errorf("permissions not extracted")
	}
	var form UpdateNodeForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	node := models.Node{InheritRoles: true}
	if err := form.Update(&node); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	if node.NodeID != 0 {
		if parent, err := v.core.Nodes.Get(int(node.NodeID)); err == nil {
			permissions = v.extendNodePermissions(c, permissions, parent)
		} else if err != sql.ErrNoRows {
			c.Logger().Error(err)
			return err
		}
	}
	if !permissions.HasPermission(models.CreateNodePermission) {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message:            "account missing permissions",
			MissingPermissions: []string{models.CreateNodePermission},
		})
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) (err error) {
			node, err = v.core.Nodes.CreateTx(tx, node, getEventOptions(c)...)
			return err
		},
	); err != nil {
		c.Logger().Error(err)
		return err
	}
	return c.JSON(http.StatusCreated, makeNode(node))
}

func (v *View) ObserveNode(c echo.Context) error {
	node, ok := c.Get(nodeKey).(models.Node)
	if !ok {
		c.Logger().Error("node not extracted")
		return fmt.Errorf("node not extracted")
	}
	return c.JSON(http.StatusOK, makeNode(node))
}

func (v *View) UpdateNode(c echo.Context) error {
	node, ok := c.Get(nodeKey).(models.Node)
	if !ok {
		return fmt.Errorf("node not extracted")
	}
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		return fmt.Errorf("permissions not extracted")
	}
	var form UpdateNodeForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := form.Update(&node); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	if node.NodeID != 0 {
		parentNode, err := v.core.Nodes.Get(int(node.NodeID))
		if err != nil {
			return err
		}
		permissions = v.extendNodePermissions(c, permissions, parentNode)
	}
	if !permissions.HasPermission(models.CreateNodePermission) {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message:            "account missing permissions",
			MissingPermissions: []string{models.CreateNodePermission},
		})
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Nodes.UpdateTx(tx, node, getEventOptions(c)...)
		},
	); err != nil {
		c.Logger().Error(err)
		return err
	}
	return c.JSON(http.StatusOK, makeNode(node))
}

func (v *View) nodeHasChildren(ctx context.Context, nodeID int) (bool, error) {
	dirs, err := v.core.Nodes.FindByDir(nodeID)
	if err != nil {
		return false, err
	}
	if len(dirs) > 0 {
		return true, nil
	}
	actions, err := v.core.Actions.FindByDir(nodeID)
	if err != nil {
		return false, err
	}
	if len(actions) > 0 {
		return true, nil
	}
	planners, err := v.core.Planners.FindByDir(nodeID)
	if err != nil {
		return false, err
	}
	if len(planners) > 0 {
		return true, nil
	}
	configs, err := v.core.Configs.FindByDir(nodeID)
	if err != nil {
		return false, err
	}
	if len(configs) > 0 {
		return true, nil
	}
	secrets, err := v.core.Secrets.FindByDir(nodeID)
	if err != nil {
		return false, err
	}
	if len(secrets) > 0 {
		return true, nil
	}
	var resources []models.Resource
	if err := v.core.WithRoTx(ctx,
		func(tx *sql.Tx) error {
			resources, err = v.core.Resources.FindByDirTx(tx, nodeID)
			return err
		},
	); err != nil {
		return false, err
	}
	if len(resources) > 0 {
		return true, nil
	}
	return false, nil
}

func (v *View) DeleteNode(c echo.Context) error {
	node, ok := c.Get(nodeKey).(models.Node)
	if !ok {
		c.Logger().Error("node not extracted")
		return fmt.Errorf("node not extracted")
	}
	hasChildren, err := v.nodeHasChildren(c.Request().Context(), node.ID)
	if err != nil {
		return err
	}
	if hasChildren {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message: "node with children can not be deleted",
		})
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Nodes.DeleteTx(tx, node.ID)
		},
	); err != nil {
		c.Logger().Error(err)
		return err
	}
	return c.JSON(http.StatusOK, makeNode(node))
}

// TODO(iudovin@): Make node permissions more logical.
func (v *View) registerNodes(g *echo.Group) {
	g.POST(
		"/v0/nodes", v.CreateNode, v.sessionAuth, v.requireAuth,
		v.requirePermission(),
	)
	g.GET(
		"/v0/nodes/:node", v.ObserveNode, v.sessionAuth, v.extractNode,
		v.requireNodePermission(models.ObserveNodePermission),
	)
	g.PATCH(
		"/v0/nodes/:node", v.UpdateNode, v.sessionAuth, v.requireAuth, v.extractNode,
		v.requireNodePermission(models.UpdateNodePermission),
	)
	g.DELETE(
		"/v0/nodes/:node", v.DeleteNode, v.sessionAuth, v.requireAuth, v.extractNode,
		v.requireNodePermission(models.DeleteNodePermission),
	)
}
