package api

import (
	"database/sql"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
)

type PlannerOld struct {
	models.Planner
	Tasks []models.Task `json:""`
}

func (v *View) GetPlannerOld(c echo.Context) error {
	planner, ok := c.Get(plannerKey).(models.Planner)
	if !ok {
		return fmt.Errorf("planner not extracted")
	}
	tasks, err := v.core.Tasks.FindByPlanner(planner.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if tasks == nil {
		tasks = make([]models.Task, 0)
	}
	return c.JSON(http.StatusOK, PlannerOld{
		Planner: planner,
		Tasks:   tasks,
	})
}

func (v *View) RunPlannerOld(c echo.Context) error {
	planner, ok := c.Get(plannerKey).(models.Planner)
	if !ok {
		return fmt.Errorf("planner not extracted")
	}
	account, ok := c.Get(authAccountKey).(models.Account)
	if !ok {
		return fmt.Errorf("account not extracted")
	}
	task := models.Task{
		ActionID:  planner.ActionID,
		Options:   planner.Options,
		PlannerID: models.NInt(planner.ID),
		OwnerID:   models.NInt(account.ID),
	}
	if err := v.core.Tasks.Create(&task); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusOK, task)
}

type Planner struct {
	ID          int                    `json:"id"`
	ActionID    int                    `json:"action_id"`
	NodeID      int                    `json:"node_id,omitempty"`
	Title       string                 `json:"title"`
	Description string                 `json:"description"`
	Options     models.TaskOptions     `json:"options"`
	Settings    models.PlannerSettings `json:"settings"`
	NextTime    int64                  `json:"next_time,omitempty"`
	CreateTime  int64                  `json:"create_time,omitempty"`
}

func makePlanner(planner models.Planner) Planner {
	return Planner{
		ID:          planner.ID,
		ActionID:    planner.ActionID,
		NodeID:      planner.DirID,
		Title:       planner.Title,
		Description: planner.Description,
		Options:     planner.Options,
		Settings:    planner.Settings,
		CreateTime:  planner.CreateTime,
	}
}

func (v *View) ObservePlanner(c echo.Context) error {
	planner, ok := c.Get(plannerKey).(models.Planner)
	if !ok {
		return fmt.Errorf("planner not extracted")
	}
	resp := makePlanner(planner)
	if state, err := v.core.PlannerStates.Get(planner.ID); err == nil {
		resp.NextTime = int64(state.NextTime)
	} else if err != sql.ErrNoRows {
		c.Logger().Error("Error:", err)
	}
	return c.JSON(http.StatusOK, resp)
}

type ObservePlannerTasksForm struct {
	Begin int `query:"begin"`
	Limit int `query:"limit"`
}

func (v *View) ObservePlannerTasks(c echo.Context) error {
	planner, ok := c.Get(plannerKey).(models.Planner)
	if !ok {
		return fmt.Errorf("planner not extracted")
	}
	form := ObservePlannerTasksForm{}
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: fmt.Sprintf("Unable to parse request: %q", err.Error()),
		})
	}
	if form.Limit <= 0 {
		form.Limit = 100
	} else if form.Limit > 1000 {
		form.Limit = 1000
	}
	tasks, nextBegin, err := v.core.Tasks.FindPageByPlanner(
		planner.ID, models.NInt(form.Begin), form.Limit,
	)
	if err != nil {
		return err
	}
	resp := TasksResponse{NextBegin: int64(nextBegin)}
	for _, task := range tasks {
		resp.Tasks = append(resp.Tasks, makeTask(task))
	}
	return c.JSON(http.StatusOK, resp)
}

type UpdatePlannerForm struct {
	NodeID      *int                    `json:"node_id"`
	Title       *string                 `json:"title"`
	Description *string                 `json:"description"`
	Options     *models.TaskOptions     `json:"options"`
	Settings    *models.PlannerSettings `json:"settings"`
}

func (f UpdatePlannerForm) Update(planner *models.Planner) *ErrorResponse {
	if f.NodeID != nil {
		planner.DirID = *f.NodeID
	}
	if f.Title != nil {
		planner.Title = *f.Title
	}
	if f.Description != nil {
		planner.Description = *f.Description
	}
	if f.Options != nil {
		planner.Options = *f.Options
	}
	if f.Settings != nil {
		planner.Settings = *f.Settings
	}
	return nil
}

func (v *View) UpdatePlanner(c echo.Context) error {
	planner, ok := c.Get(plannerKey).(models.Planner)
	if !ok {
		return fmt.Errorf("planner not extracted")
	}
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		return fmt.Errorf("permissions not extracted")
	}
	var form UpdatePlannerForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := form.Update(&planner); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	if planner.DirID != 0 {
		node, err := v.core.Nodes.Get(int(planner.DirID))
		if err != nil {
			return err
		}
		permissions = v.extendNodePermissions(c, permissions, node)
	}
	if !permissions.HasPermission(models.CreatePlannerPermission) {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message:            "account missing permissions",
			MissingPermissions: []string{models.CreatePlannerPermission},
		})
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) (err error) {
			return v.core.Planners.UpdateTx(tx, planner, getEventOptions(c)...)
		},
	); err != nil {
		c.Logger().Error(err)
		return err
	}
	resp := makePlanner(planner)
	if state, err := v.core.PlannerStates.Get(planner.ID); err == nil {
		resp.NextTime = int64(state.NextTime)
	} else if err != sql.ErrNoRows {
		c.Logger().Error("Error:", err)
	}
	return c.JSON(http.StatusOK, resp)
}

func (v *View) RunPlanner(c echo.Context) error {
	planner, ok := c.Get(plannerKey).(models.Planner)
	if !ok {
		return fmt.Errorf("planner not extracted")
	}
	account, ok := c.Get(authAccountKey).(models.Account)
	if !ok {
		return fmt.Errorf("account not extracted")
	}
	task := models.Task{
		ActionID:  planner.ActionID,
		PlannerID: models.NInt(planner.ID),
		OwnerID:   models.NInt(account.ID),
		Options:   planner.Options,
	}
	if err := v.core.Tasks.Create(&task); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusOK, makeTask(task))
}

func (v *View) EnablePlanner(c echo.Context) error {
	planner, ok := c.Get(plannerKey).(models.Planner)
	if !ok {
		return fmt.Errorf("planner not extracted")
	}
	planner.Settings.Enabled = true
	state := models.PlannerState{
		PlannerID: planner.ID,
		NextTime:  models.NInt64(planner.Next(time.Now().Unix())),
	}
	if err := gosql.WithTxContext(c.Request().Context(), v.core.DB, nil, func(tx *sql.Tx) error {
		if err := v.core.Planners.UpdateTx(tx, planner, getEventOptions(c)...); err != nil {
			return err
		}
		if err := v.core.PlannerStates.UpsertTx(tx, state, "next_time"); err != nil {
			return err
		}
		return nil
	}); err != nil {
		return err
	}
	resp := makePlanner(planner)
	if state, err := v.core.PlannerStates.Get(planner.ID); err == nil {
		resp.NextTime = int64(state.NextTime)
	} else if err != sql.ErrNoRows {
		c.Logger().Error("Error:", err)
	}
	return c.JSON(http.StatusOK, resp)
}

func (v *View) DisablePlanner(c echo.Context) error {
	planner, ok := c.Get(plannerKey).(models.Planner)
	if !ok {
		return fmt.Errorf("planner not extracted")
	}
	planner.Settings.Enabled = false
	state := models.PlannerState{
		PlannerID: planner.ID,
		NextTime:  0,
	}
	if err := gosql.WithTxContext(c.Request().Context(), v.core.DB, nil, func(tx *sql.Tx) error {
		if err := v.core.Planners.UpdateTx(tx, planner, getEventOptions(c)...); err != nil {
			return err
		}
		if err := v.core.PlannerStates.UpsertTx(tx, state, "next_time"); err != nil {
			return err
		}
		return nil
	}); err != nil {
		return err
	}
	resp := makePlanner(planner)
	if state, err := v.core.PlannerStates.Get(planner.ID); err == nil {
		resp.NextTime = int64(state.NextTime)
	} else if err != sql.ErrNoRows {
		c.Logger().Error("Error:", err)
	}
	return c.JSON(http.StatusOK, resp)
}

func (v *View) registerPlanners(g *echo.Group) {
	g.GET(
		"/v0/planners/:planner", v.ObservePlanner, v.sessionAuth, v.extractPlanner,
		v.requireNodePermission(models.ObservePlannerPermission),
	)
	g.PATCH(
		"/v0/planners/:planner", v.UpdatePlanner, v.sessionAuth, v.requireAuth, v.extractPlanner,
		v.requireNodePermission(models.UpdatePlannerPermission),
	)
	g.POST(
		"/v0/planners/:planner/enable", v.EnablePlanner, v.sessionAuth, v.requireAuth, v.extractPlanner,
		v.requireNodePermission(models.UpdatePlannerPermission),
	)
	g.POST(
		"/v0/planners/:planner/disable", v.DisablePlanner, v.sessionAuth, v.requireAuth, v.extractPlanner,
		v.requireNodePermission(models.UpdatePlannerPermission),
	)
	g.POST(
		"/v0/planners/:planner/run", v.RunPlanner, v.sessionAuth, v.requireAuth, v.extractPlanner,
		v.requireNodePermission(models.RunPlannerPermission),
	)
	g.GET(
		"/v0/planners/:planner/tasks", v.ObservePlannerTasks, v.sessionAuth, v.extractPlanner,
		v.requireNodePermission(models.ObservePlannerPermission, models.ObserveTaskPermission),
	)
	// Deprecated.
	g.GET(
		"/planners/:planner", v.GetPlannerOld, v.sessionAuth, v.extractPlanner,
		v.requireNodePermission(models.ObservePlannerPermission),
	)
	g.POST(
		"/planners/:planner", v.RunPlannerOld, v.sessionAuth, v.requireAuth, v.extractPlanner,
		v.requireNodePermission(models.RunPlannerPermission),
	)
}

func (v *View) extractPlanner(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		id, err := strconv.Atoi(c.Param("planner"))
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusBadRequest)
		}
		planner, err := v.core.Planners.Get(id)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		c.Set(plannerKey, planner)
		if planner.DirID != 0 {
			node, err := v.core.Nodes.Get(int(planner.DirID))
			if err != nil {
				if err == sql.ErrNoRows {
					return c.NoContent(http.StatusNotFound)
				}
				c.Logger().Error(err)
				return err
			}
			c.Set(nodeKey, node)
		}
		return next(c)
	}
}
