package api

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/models"
)

const (
	ContentTypeMeta = "@ContentType"
	DisplayTypeMeta = "@DisplayType"
)

func (v *View) GetResource(c echo.Context) error {
	resource, ok := c.Get(resourceKey).(models.Resource)
	if !ok {
		return fmt.Errorf("resource not extracted")
	}
	return c.JSON(http.StatusOK, resource)
}

func (v *View) GetResourceData(c echo.Context) error {
	resource, ok := c.Get(resourceKey).(models.Resource)
	if !ok {
		return fmt.Errorf("resource not extracted")
	}
	reader, writer := io.Pipe()
	go func() {
		// Store error that should be returned by writer.
		var errWriter error
		// We should always close writer.
		defer func() {
			// We should handle panics to not totally fail.
			if r := recover(); r != nil {
				c.Logger().Error(r)
				// Try to cast recovered value to error.
				if err, ok := r.(error); ok {
					errWriter = err
				} else {
					// Writer error should not be nil.
					errWriter = fmt.Errorf("panic: %v", r)
				}
			}
			// CloseWithError always returns nil.
			_ = writer.CloseWithError(errWriter)
		}()
		// Set writer error with download status.
		errWriter = v.core.Resources.Download(resource, writer)
	}()
	c.Response().Header().Set(
		echo.HeaderContentDisposition,
		fmt.Sprintf("attachment; filename=%q", resource.Title),
	)
	if contentType, ok := resource.ContentTypeMeta(); ok {
		return c.Stream(http.StatusOK, contentType, reader)
	}
	return c.Stream(http.StatusOK, echo.MIMEOctetStream, reader)
}

func (v *View) registerResources(g *echo.Group) {
	g.POST(
		"/resources", v.CreateResource, v.sessionAuth, v.requireAuth,
		v.requirePermission(models.LegacyAuthPermission),
	)
	g.GET(
		"/resources/:resource",
		v.GetResource, v.sessionAuth, v.extractResource,
		v.requirePermission(models.LegacyAuthPermission),
	)
	g.GET(
		"/resources/:resource/data",
		v.GetResourceData, v.sessionAuth, v.extractResource,
		v.requirePermission(models.LegacyAuthPermission),
	)
}

type ResourceForm struct {
	DirID *int   `json:"" query:"DirID"`
	Title string `json:"" query:"Title"`
	Meta  string `json:"" query:"Meta"`
}

func (f *ResourceForm) Update(resource *models.Resource) {
	if f.DirID != nil {
		resource.DirID = models.NInt(*f.DirID)
	}
	if f.Title != "" {
		resource.Title = f.Title
	}
	if f.Meta != "" {
		if err := json.Unmarshal([]byte(f.Meta), &resource.Meta); err != nil {
			resource.Meta = nil
		}
	}
}

func (v *View) CreateResource(c echo.Context) error {
	var form ResourceForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.NoContent(http.StatusBadRequest)
	}
	var resource models.Resource
	form.Update(&resource)
	if task, ok := c.Get(authTaskKey).(models.Task); ok {
		resource.TaskID = models.NInt64(task.ID)
	}
	if err := v.core.Resources.Validate(resource); err != nil {
		if err, ok := err.(models.FieldListError); ok {
			c.Logger().Warn(err)
			return c.JSON(http.StatusBadRequest, ErrorResponse{
				FieldErrors: err,
			})
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := resource.GenerateKey(); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := resource.GenerateSecret(); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	file, err := c.FormFile("Data")
	if err != nil {
		c.Logger().Warn(err)
		return c.NoContent(http.StatusBadRequest)
	}
	resource.SetSize(file.Size)
	reader, err := file.Open()
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	defer func() {
		_ = reader.Close()
	}()
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) (err error) {
			resource, err = v.core.Resources.CreateTx(
				tx, resource, getEventOptions(c)...,
			)
			return
		},
	); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := v.core.Resources.Upload(resource, reader); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusOK, resource)
}

func (v *View) extractResource(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		id, err := strconv.ParseInt(c.Param("resource"), 10, 64)
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusBadRequest)
		}
		var resource models.Resource
		if err := v.core.WithRoTx(
			c.Request().Context(),
			func(tx *sql.Tx) error {
				resource, err = v.core.Resources.GetTx(tx, id)
				return err
			},
		); err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		c.Set(resourceKey, resource)
		return next(c)
	}
}
