package api

import (
	"database/sql"
	"fmt"
	"net/http"
	"sort"
	"strconv"

	"a.yandex-team.ru/drive/runner/models"
	"github.com/labstack/echo/v4"
)

// Role represents role.
type Role struct {
	ID          int      `json:"id"`
	Name        string   `json:"name"`
	Permissions []string `json:"permissions,omitempty"`
}

// Roles represents roles response.
type Roles struct {
	// Roles contains list of roles.
	Roles []Role `json:"roles"`
}

// ObserveRoles returns list of roles.
func (v *View) ObserveRoles(c echo.Context) error {
	roles, err := v.core.Roles.All()
	if err != nil {
		return err
	}
	var resp Roles
	for _, role := range roles {
		respRole := Role{
			ID:   role.ID,
			Name: role.Name,
		}
		permissions, err := role.GetPermissions()
		if err != nil {
			c.Logger().Error(err)
		} else {
			respRole.Permissions = permissions
		}
		resp.Roles = append(resp.Roles, respRole)
	}
	sort.Sort(roleSorter(resp.Roles))
	return c.JSON(http.StatusOK, resp)
}

type UpdateRoleForm struct {
	Name        *string  `json:"name"`
	Permissions []string `json:"permissions"`
}

func (f UpdateRoleForm) Update(role *models.Role) *ErrorResponse {
	if f.Name != nil {
		role.Name = *f.Name
	}
	if f.Permissions != nil {
		if err := role.SetPermissions(f.Permissions); err != nil {
			return &ErrorResponse{
				Message: "unable to set role permissions",
			}
		}
	}
	resp := ErrorResponse{
		Message:       "invalid role",
		InvalidFields: InvalidFields{},
	}
	if len(role.Name) == 0 {
		resp.InvalidFields["name"] = InvalidField{
			Message: "name can not be empty",
		}
	}
	var invalidPermissions []string
	for _, name := range f.Permissions {
		if !models.HasPermission(name) {
			invalidPermissions = append(invalidPermissions, name)
		}
	}
	if len(invalidPermissions) > 0 {
		resp.InvalidFields["permissions"] = InvalidField{
			Message: "permissions has invalid permission names",
		}
	}
	if len(resp.InvalidFields) > 0 {
		return &resp
	}
	return nil
}

func (v *View) CreateRole(c echo.Context) error {
	var form UpdateRoleForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	var role models.Role
	if err := form.Update(&role); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := v.core.Roles.CreateTx(
		v.core.DB, &role, getEventOptions(c)...,
	); err != nil {
		c.Logger().Error(err)
		return err
	}
	resp := Role{
		ID:   role.ID,
		Name: role.Name,
	}
	permissions, err := role.GetPermissions()
	if err != nil {
		c.Logger().Error(err)
	} else {
		resp.Permissions = permissions
	}
	return c.JSON(http.StatusCreated, resp)
}

func (v *View) UpdateRole(c echo.Context) error {
	var form UpdateRoleForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	role, ok := c.Get(roleKey).(models.Role)
	if !ok {
		return fmt.Errorf("role not extracted")
	}
	if err := form.Update(&role); err != nil {
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) (err error) {
			return v.core.Roles.UpdateTx(
				tx, role, getEventOptions(c)...,
			)
		},
	); err != nil {
		c.Logger().Error(err)
		return err
	}
	resp := Role{
		ID:   role.ID,
		Name: role.Name,
	}
	permissions, err := role.GetPermissions()
	if err != nil {
		c.Logger().Error(err)
	} else {
		resp.Permissions = permissions
	}
	return c.JSON(http.StatusOK, resp)
}

func (v *View) DeleteRole(c echo.Context) error {
	role, ok := c.Get(roleKey).(models.Role)
	if !ok {
		return fmt.Errorf("role not extracted")
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) (err error) {
			return v.core.Roles.RemoveTx(
				tx, role.ID, getEventOptions(c)...,
			)
		},
	); err != nil {
		return err
	}
	resp := Role{
		ID:   role.ID,
		Name: role.Name,
	}
	permissions, err := role.GetPermissions()
	if err != nil {
		c.Logger().Error(err)
	} else {
		resp.Permissions = permissions
	}
	return c.JSON(http.StatusOK, resp)
}

type AccountRole struct {
	ID   int    `json:"id"`
	Name string `json:"name,omitempty"`
}

type AccountRoles struct {
	Roles []AccountRole `json:"roles"`
}

func (v *View) ObserveAccountRoles(c echo.Context) error {
	account, ok := c.Get(accountKey).(models.Account)
	if !ok {
		return fmt.Errorf("account not extracted")
	}
	accountRoles, err := v.core.AccountRoles.FindByAccount(account.ID)
	if err != nil {
		return err
	}
	var resp AccountRoles
	for _, accountRole := range accountRoles {
		role, err := v.core.Roles.Get(accountRole.RoleID)
		if err != nil {
			return err
		}
		resp.Roles = append(resp.Roles, AccountRole{
			ID:   accountRole.ID,
			Name: role.Name,
		})
	}
	return c.JSON(http.StatusOK, resp)
}

type CreateAccountRoleForm struct {
	Name string `json:"name" form:"name"`
}

func (v *View) CreateAccountRole(c echo.Context) error {
	account, ok := c.Get(accountKey).(models.Account)
	if !ok {
		return fmt.Errorf("account not extracted")
	}
	var form CreateAccountRoleForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	role, err := v.core.Roles.GetByName(form.Name)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.JSON(http.StatusNotFound, ErrorResponse{
				Message: fmt.Sprintf("Role %q not found", form.Name),
			})
		}
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	accountRole := models.AccountRole{
		AccountID: account.ID,
		RoleID:    role.ID,
	}
	if err := v.core.AccountRoles.CreateTx(
		v.core.DB, &accountRole, getEventOptions(c)...,
	); err != nil {
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	resp := AccountRole{ID: accountRole.ID, Name: role.Name}
	return c.JSON(http.StatusCreated, resp)
}

func (v *View) DeleteAccountRole(c echo.Context) error {
	account, ok := c.Get(accountKey).(models.Account)
	if !ok {
		return fmt.Errorf("account not extracted")
	}
	accountRole, ok := c.Get(accountRoleKey).(models.AccountRole)
	if !ok {
		return fmt.Errorf("account role not extracted")
	}
	if accountRole.AccountID != account.ID {
		return c.JSON(http.StatusNotFound, ErrorResponse{
			Message: "role does not exists",
		})
	}
	if err := v.core.AccountRoles.RemoveTx(
		v.core.DB, accountRole.ID, getEventOptions(c)...,
	); err != nil {
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	resp := AccountRole{ID: accountRole.ID}
	return c.JSON(http.StatusOK, resp)
}

type NodeRole struct {
	ID        int    `json:"id"`
	AccountID int    `json:"account_id,omitempty"`
	Name      string `json:"name,omitempty"`
}

type NodeRoles struct {
	Roles []NodeRole `json:"roles"`
}

func (v *View) ObserveNodeRoles(c echo.Context) error {
	node, ok := c.Get(nodeKey).(models.Node)
	if !ok {
		return fmt.Errorf("node not extracted")
	}
	nodeRoles, err := v.core.AccountNodeRoles.FindByNode(node.ID)
	if err != nil {
		return err
	}
	var resp NodeRoles
	for _, nodeRole := range nodeRoles {
		role, err := v.core.Roles.Get(nodeRole.RoleID)
		if err != nil {
			return err
		}
		resp.Roles = append(resp.Roles, NodeRole{
			ID:        nodeRole.ID,
			AccountID: nodeRole.AccountID,
			Name:      role.Name,
		})
	}
	return c.JSON(http.StatusOK, resp)
}

type CreateNodeRoleForm struct {
	AccountID int    `json:"account_id" form:"account_id"`
	Name      string `json:"name" form:"name"`
}

func (v *View) CreateNodeRole(c echo.Context) error {
	node, ok := c.Get(nodeKey).(models.Node)
	if !ok {
		return fmt.Errorf("node not extracted")
	}
	var form CreateNodeRoleForm
	if err := c.Bind(&form); err != nil {
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	account, err := v.core.Accounts.Get(form.AccountID)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.JSON(http.StatusNotFound, ErrorResponse{
				Message: fmt.Sprintf("Account %d not found", form.AccountID),
			})
		}
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	role, err := v.core.Roles.GetByName(form.Name)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.JSON(http.StatusNotFound, ErrorResponse{
				Message: fmt.Sprintf("Role %q not found", form.Name),
			})
		}
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	nodeRole := models.AccountNodeRole{
		AccountID: account.ID,
		NodeID:    node.ID,
		RoleID:    role.ID,
	}
	if err := v.core.AccountNodeRoles.CreateTx(
		v.core.DB, &nodeRole, getEventOptions(c)...,
	); err != nil {
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	resp := NodeRole{
		ID:        nodeRole.ID,
		AccountID: nodeRole.AccountID,
		Name:      role.Name,
	}
	return c.JSON(http.StatusCreated, resp)
}

func (v *View) DeleteNodeRole(c echo.Context) error {
	node, ok := c.Get(nodeKey).(models.Node)
	if !ok {
		return fmt.Errorf("node not extracted")
	}
	nodeRole, ok := c.Get(accountNodeRoleKey).(models.AccountNodeRole)
	if !ok {
		return fmt.Errorf("account node role not extracted")
	}
	if nodeRole.NodeID != node.ID {
		return c.JSON(http.StatusNotFound, ErrorResponse{
			Message: "role does not exists",
		})
	}
	if err := v.core.AccountNodeRoles.RemoveTx(
		v.core.DB, nodeRole.ID, getEventOptions(c)...,
	); err != nil {
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	resp := NodeRole{ID: nodeRole.ID}
	return c.JSON(http.StatusOK, resp)
}

func (v *View) registerRoles(g *echo.Group) {
	g.GET(
		"/v0/roles", v.ObserveRoles, v.sessionAuth,
		v.requirePermission(models.ObserveRolePermission),
	)
	g.POST(
		"/v0/roles", v.CreateRole, v.sessionAuth, v.requireAuth,
		v.requirePermission(models.CreateRolePermission),
	)
	g.PATCH(
		"/v0/roles/:role", v.UpdateRole, v.sessionAuth, v.requireAuth, v.extractRole,
		v.requirePermission(models.UpdateRolePermission),
	)
	g.DELETE(
		"/v0/roles/:role", v.DeleteRole, v.sessionAuth, v.requireAuth, v.extractRole,
		v.requirePermission(models.DeleteRolePermission),
	)
	g.GET(
		"/v0/accounts/:account/roles", v.ObserveAccountRoles, v.sessionAuth, v.extractAccount,
		v.requirePermission(models.ObserveAccountRolePermission),
	)
	g.POST(
		"/v0/accounts/:account/roles", v.CreateAccountRole, v.sessionAuth, v.requireAuth, v.extractAccount,
		v.requirePermission(models.CreateAccountRolePermission),
	)
	g.DELETE(
		"/v0/accounts/:account/roles/:account_role", v.DeleteAccountRole, v.sessionAuth, v.requireAuth,
		v.extractAccount, v.extractAccountRole,
		v.requirePermission(models.DeleteAccountRolePermission),
	)
	g.GET(
		"/v0/nodes/:node/roles", v.ObserveNodeRoles, v.sessionAuth, v.extractNode,
		v.requireNodePermission(models.ObserveNodeRolePermission),
	)
	g.POST(
		"/v0/nodes/:node/roles", v.CreateNodeRole, v.sessionAuth, v.requireAuth, v.extractNode,
		v.requireNodePermission(models.CreateNodeRolePermission),
	)
	g.DELETE(
		"/v0/nodes/:node/roles/:account_node_role", v.DeleteNodeRole, v.sessionAuth, v.requireAuth,
		v.extractNode, v.extractAccountNodeRole,
		v.requireNodePermission(models.DeleteNodeRolePermission),
	)
}

func (v *View) registerSocketRoles(g *echo.Group) {
	g.GET("/v0/roles", v.ObserveRoles)
	g.POST("/v0/roles", v.CreateRole)
	g.PATCH("/v0/roles/:role", v.UpdateRole, v.extractRole)
	g.DELETE("/v0/roles/:role", v.DeleteRole, v.extractRole)
	g.GET(
		"/v0/accounts/:account/roles",
		v.ObserveAccountRoles, v.extractAccount,
	)
	g.POST(
		"/v0/accounts/:account/roles", v.CreateAccountRole,
		v.extractAccount,
	)
	g.DELETE(
		"/v0/accounts/:account/roles/:account_role", v.DeleteAccountRole,
		v.extractAccount, v.extractAccountRole,
	)
	g.GET(
		"/v0/nodes/:node/roles", v.ObserveNodeRoles,
		v.extractNode,
	)
	g.POST(
		"/v0/nodes/:node/roles", v.CreateNodeRole,
		v.extractNode,
	)
	g.DELETE(
		"/v0/nodes/:node/roles/:account_node_role", v.DeleteNodeRole,
		v.extractNode, v.extractAccountNodeRole,
	)
}

type roleSorter []Role

func (a roleSorter) Len() int {
	return len(a)
}

func (a roleSorter) Swap(i, j int) {
	a[i], a[j] = a[j], a[i]
}

func (a roleSorter) Less(i, j int) bool {
	return a[i].Name < a[j].Name
}

func (v *View) extractRole(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		param := c.Param("role")
		id, err := strconv.Atoi(param)
		if err != nil {
			role, err := v.core.Roles.GetByName(param)
			if err != nil {
				if err == sql.ErrNoRows {
					return c.NoContent(http.StatusNotFound)
				}
				c.Logger().Error(err)
				return c.NoContent(http.StatusInternalServerError)
			}
			c.Set(roleKey, role)
			return next(c)
		}
		role, err := v.core.Roles.Get(id)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		c.Set(roleKey, role)
		return next(c)
	}
}

func (v *View) extractAccount(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		id, err := strconv.Atoi(c.Param("account"))
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusBadRequest)
		}
		account, err := v.core.Accounts.Get(id)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return err
		}
		c.Set(accountKey, account)
		return next(c)
	}
}

func (v *View) extractStaffAccount(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		login := c.Param("user")
		accounts, err := v.core.Accounts.FindByLogin(login)
		if err != nil {
			c.Logger().Error(err)
			return err
		}
		pos := -1
		for i := range accounts {
			if accounts[i].Kind == models.StaffAccount {
				pos = i
				break
			}
		}
		if pos == -1 {
			return fmt.Errorf("user not found")
		}
		c.Set(accountKey, accounts[pos])
		return next(c)
	}
}

func (v *View) extractAccountRole(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		param := c.Param("account_role")
		id, err := strconv.Atoi(param)
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusBadRequest)
		}
		role, err := v.core.AccountRoles.Get(id)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		c.Set(accountRoleKey, role)
		return next(c)
	}
}

func (v *View) extractAccountNodeRole(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		param := c.Param("account_node_role")
		id, err := strconv.Atoi(param)
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusBadRequest)
		}
		role, err := v.core.AccountNodeRoles.Get(id)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		c.Set(accountNodeRoleKey, role)
		return next(c)
	}
}
