package api

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
)

type SearchForm struct {
	Query string `query:"q"`
	Limit int    `query:"limit"`
}

func (v *View) Search(c echo.Context) error {
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		return fmt.Errorf("permissions not extracted")
	}
	form := SearchForm{Limit: 25}
	if err := c.Bind(&form); err != nil {
		return c.NoContent(http.StatusBadRequest)
	}
	actions, err := v.core.Actions.FindByQuery(form.Query)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if actions == nil {
		actions = []models.Action{}
	}
	if newLen := 0; len(actions) > 0 {
		for i := 0; i < len(actions) && newLen < form.Limit; i++ {
			action := actions[i]
			node, err := v.core.Nodes.Get(int(action.DirID))
			if err != nil {
				continue
			}
			nodePermissions := v.extendNodePermissions(c, permissions, node)
			if nodePermissions.HasPermission(models.ObserveActionPermission) {
				actions[newLen] = action
				newLen++
			}
		}
		actions = actions[:newLen]
		form.Limit -= newLen
	}
	planners, err := v.core.Planners.FindByQuery(form.Query)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if planners == nil {
		planners = []models.Planner{}
	}
	if newLen := 0; len(planners) > 0 {
		for i := 0; i < len(planners) && newLen < form.Limit; i++ {
			planner := planners[i]
			node, err := v.core.Nodes.Get(int(planner.DirID))
			if err != nil {
				continue
			}
			nodePermissions := v.extendNodePermissions(c, permissions, node)
			if nodePermissions.HasPermission(models.ObservePlannerPermission) {
				planners[newLen] = planner
				newLen++
			}
		}
		planners = planners[:newLen]
		form.Limit -= newLen
	}
	return c.JSON(http.StatusOK, TreeList{
		Actions:  actions,
		Planners: planners,
	})
}
