package api

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
)

type SecretForm struct {
	DirID *int              `json:""`
	Title string            `json:""`
	Type  models.SecretType `json:""`
	Data  json.RawMessage   `json:""`
}

func (f *SecretForm) Update(secret *models.Secret) {
	if f.DirID != nil {
		secret.DirID = models.NInt(*f.DirID)
	}
	if f.Title != "" {
		secret.Title = f.Title
	}
	if f.Type != "" {
		secret.Type = f.Type
	}
	if f.Data != nil {
		secret.Data = models.JSON(f.Data)
	}
}

func (v *View) GetSecret(c echo.Context) error {
	secret, ok := c.Get(secretKey).(models.Secret)
	if !ok {
		return fmt.Errorf("secret not extracted")
	}
	return c.JSON(http.StatusOK, secret)
}

func (v *View) CreateSecretOld(c echo.Context) error {
	var form SecretForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.NoContent(http.StatusBadRequest)
	}
	var secret models.Secret
	form.Update(&secret)
	if err := v.core.Secrets.Validate(secret); err != nil {
		if err, ok := err.(models.FieldListError); ok {
			c.Logger().Warn(err)
			return c.JSON(http.StatusBadRequest, ErrorResponse{
				FieldErrors: err,
			})
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Secrets.CreateTx(tx, &secret, getEventOptions(c)...)
		},
	); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusCreated, secret)
}

// UpdateSecret updates specified secret.
func (v *View) UpdateSecretOld(c echo.Context) error {
	var form SecretForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.NoContent(http.StatusBadRequest)
	}
	secret, ok := c.Get(secretKey).(models.Secret)
	if !ok {
		return fmt.Errorf("secret not extracted")
	}
	form.Update(&secret)
	if err := v.core.Secrets.Validate(secret); err != nil {
		if err, ok := err.(models.FieldListError); ok {
			c.Logger().Warn(err)
			return c.JSON(http.StatusBadRequest, ErrorResponse{
				FieldErrors: err,
			})
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Secrets.UpdateTx(tx, secret, getEventOptions(c)...)
		},
	); err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusOK, secret)
}

// RemoveSecret removes specified secret from database.
func (v *View) RemoveSecret(c echo.Context) error {
	secret, ok := c.Get(secretKey).(models.Secret)
	if !ok {
		return fmt.Errorf("secret not extracted")
	}
	if err := v.core.WithTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			return v.core.Secrets.RemoveTx(tx, secret.ID, getEventOptions(c)...)
		},
	); err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.NoContent(http.StatusOK)
}

type Secret struct {
	ID     int               `json:"id"`
	NodeID int               `json:"node_id,omitempty"`
	Title  string            `json:"title"`
	Kind   models.SecretType `json:"kind"`
	Data   models.JSON       `json:"data"`
}

func (v *View) ObserveSecret(c echo.Context) error {
	secret, ok := c.Get(secretKey).(models.Secret)
	if !ok {
		return fmt.Errorf("secret not extracted")
	}
	return c.JSON(http.StatusOK, makeSecret(secret))
}

type UpdateSecretForm struct {
	NodeID *int               `json:"node_id"`
	Title  *string            `json:"title"`
	Kind   *models.SecretType `json:"kind"`
	Data   *json.RawMessage   `json:"data"`
}

func (f UpdateSecretForm) Update(secret *models.Secret) *ErrorResponse {
	if f.NodeID != nil {
		secret.DirID = models.NInt(*f.NodeID)
	}
	if f.Title != nil {
		secret.Title = *f.Title
	}
	if f.Kind != nil {
		secret.Type = *f.Kind
	}
	if f.Data != nil {
		secret.Data = models.JSON(*f.Data)
	}
	return nil
}

func makeSecret(secret models.Secret) Secret {
	return Secret{
		ID:     secret.ID,
		NodeID: int(secret.DirID),
		Title:  secret.Title,
		Kind:   secret.Type,
		Data:   secret.Data,
	}
}

func (v *View) CreateSecret(c echo.Context) error {
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		c.Logger().Error("permissions not extracted")
		return fmt.Errorf("permissions not extracted")
	}
	var form UpdateSecretForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	secret := models.Secret{}
	if err := form.Update(&secret); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := v.core.Secrets.Validate(secret); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	if secret.DirID != 0 {
		if parent, err := v.core.Nodes.Get(int(secret.DirID)); err == nil {
			permissions = v.extendNodePermissions(c, permissions, parent)
		} else if err != sql.ErrNoRows {
			c.Logger().Error(err)
			return err
		}
	}
	if !permissions.HasPermission(models.CreateSecretPermission) {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message:            "account missing permissions",
			MissingPermissions: []string{models.CreateSecretPermission},
		})
	}
	if err := v.core.Secrets.CreateTx(
		v.core.DB, &secret, getEventOptions(c)...,
	); err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	return c.JSON(http.StatusCreated, makeSecret(secret))
}

func (v *View) UpdateSecret(c echo.Context) error {
	secret, ok := c.Get(secretKey).(models.Secret)
	if !ok {
		return fmt.Errorf("secret not extracted")
	}
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		return fmt.Errorf("permissions not extracted")
	}
	var form UpdateSecretForm
	if err := c.Bind(&form); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	if err := form.Update(&secret); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, err)
	}
	if err := v.core.Secrets.Validate(secret); err != nil {
		c.Logger().Warn(err)
		return c.JSON(http.StatusBadRequest, ErrorResponse{
			Message: err.Error(),
		})
	}
	if secret.DirID != 0 {
		node, err := v.core.Nodes.Get(int(secret.DirID))
		if err != nil {
			return err
		}
		permissions = v.extendNodePermissions(c, permissions, node)
	}
	if !permissions.HasPermission(models.CreateSecretPermission) {
		return c.JSON(http.StatusForbidden, ErrorResponse{
			Message:            "account missing permissions",
			MissingPermissions: []string{models.CreateSecretPermission},
		})
	}
	if err := v.core.Secrets.UpdateTx(
		v.core.DB, secret, getEventOptions(c)...,
	); err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	return c.JSON(http.StatusOK, makeSecret(secret))
}

func (v *View) DeleteSecret(c echo.Context) error {
	secret, ok := c.Get(secretKey).(models.Secret)
	if !ok {
		return fmt.Errorf("secret not extracted")
	}
	if err := v.core.Secrets.RemoveTx(
		v.core.DB, secret.ID, getEventOptions(c)...,
	); err != nil {
		c.Logger().Error(err)
		return c.JSON(http.StatusInternalServerError, ErrorResponse{
			Message: err.Error(),
		})
	}
	return c.JSON(http.StatusOK, makeSecret(secret))
}

func (v *View) registerSecrets(g *echo.Group) {
	g.GET(
		"/v0/secrets/:secret", v.ObserveSecret, v.sessionAuth, v.extractSecret,
		v.requireNodePermission(models.ObserveSecretPermission),
	)
	g.POST(
		"/v0/secrets", v.CreateSecret, v.sessionAuth, v.requireAuth,
		v.requirePermission(),
	)
	g.PATCH(
		"/v0/secrets/:secret", v.UpdateSecret, v.sessionAuth, v.requireAuth, v.extractSecret,
		v.requireNodePermission(models.UpdateSecretPermission),
	)
	g.DELETE(
		"/v0/secrets/:secret", v.DeleteSecret, v.sessionAuth, v.requireAuth, v.extractSecret,
		v.requireNodePermission(models.DeleteSecretPermission),
	)
	// Deprecated.
	g.GET(
		"/secrets/:secret", v.GetSecret, v.sessionAuth, v.extractSecret,
		v.requireNodePermission(models.ObserveSecretPermission),
	)
	g.POST(
		"/secrets", v.CreateSecretOld, v.sessionAuth, v.requireAuth,
		v.requirePermission(models.LegacyAuthPermission),
	)
	g.PATCH(
		"/secrets/:secret", v.UpdateSecretOld, v.sessionAuth, v.requireAuth, v.extractSecret,
		v.requireNodePermission(models.UpdateSecretPermission),
	)
	g.DELETE(
		"/secrets/:secret", v.RemoveSecret, v.sessionAuth, v.requireAuth, v.extractSecret,
		v.requireNodePermission(models.DeleteSecretPermission),
	)
}

func (v *View) extractSecret(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		id, err := strconv.Atoi(c.Param("secret"))
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusBadRequest)
		}
		secret, err := v.core.Secrets.Get(id)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		c.Set(secretKey, secret)
		if secret.DirID != 0 {
			node, err := v.core.Nodes.Get(int(secret.DirID))
			if err != nil {
				if err == sql.ErrNoRows {
					return c.NoContent(http.StatusNotFound)
				}
				c.Logger().Error(err)
				return err
			}
			c.Set(nodeKey, node)
		}
		return next(c)
	}
}
