package api

import (
	"database/sql"
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/models"
)

type TaskOld struct {
	models.Task
	Action    models.Action     `json:""`
	Resources []models.Resource `json:""`
}

type TaskPager struct {
	Items   []models.Task `json:""`
	Page    int           `json:""`
	EndPage int           `json:""`
	PerPage int           `json:""`
}

const (
	PageParam    = "Page"
	PerPageParam = "PerPage"
)

func (v *View) GetTasks(c echo.Context) error {
	pager := TaskPager{
		PerPage: getIntQueryParam(c, PerPageParam, 100),
	}
	account := c.Get(authAccountKey).(models.Account)
	if account.ID > 0 {
		pager.Page = getIntQueryParam(c, PageParam, 0)
		var err error
		var tasks []models.Task
		tasks, pager.EndPage, err = v.core.Tasks.ListByUser(
			account.ID, pager.Page, pager.PerPage,
		)
		if err != nil {
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		pager.Items = tasks
		if pager.Items == nil {
			pager.Items = []models.Task{}
		}
		pager.EndPage = (pager.EndPage + pager.PerPage - 1) / pager.PerPage
	}
	return c.JSON(http.StatusOK, pager)
}

func (v *View) GetTask(c echo.Context) error {
	task, err := v.getTaskByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	action, err := v.core.Actions.Get(task.ActionID)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	var resources []models.Resource
	if err := v.core.WithRoTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			resources, err = v.core.Resources.FindByTaskTx(tx, task.ID)
			return err
		},
	); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if resources == nil {
		resources = []models.Resource{}
	}
	return c.JSON(http.StatusOK, TaskOld{
		Task:      task,
		Action:    action,
		Resources: resources,
	})
}

func (v *View) GetTaskSystemLogs(c echo.Context) error {
	task, err := v.getTaskByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	logs, err := v.core.TaskLogs.FindByTask(task.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	var builder strings.Builder
	for _, log := range logs {
		if log.Type == models.SystemTaskLog {
			builder.WriteString(log.Lines)
			builder.WriteString("\n")
		}
	}
	return c.String(http.StatusOK, builder.String())
}

func (v *View) GetTaskStdoutLogs(c echo.Context) error {
	task, err := v.getTaskByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	logs, err := v.core.TaskLogs.FindByTask(task.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	var builder strings.Builder
	for _, log := range logs {
		if log.Type == models.StdoutTaskLog {
			builder.WriteString(log.Lines)
			builder.WriteString("\n")
		}
	}
	return c.String(http.StatusOK, builder.String())
}

func (v *View) GetTaskStderrLogs(c echo.Context) error {
	task, err := v.getTaskByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	logs, err := v.core.TaskLogs.FindByTask(task.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	var builder strings.Builder
	for _, log := range logs {
		if log.Type == models.StderrTaskLog {
			builder.WriteString(log.Lines)
		}
	}
	return c.String(http.StatusOK, builder.String())
}

func (v *View) AbortTaskOld(c echo.Context) error {
	task, err := v.getTaskByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	task.Status = models.AbortingTask
	if err := v.core.Tasks.Update(&task); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusOK, task)
}

func (v *View) getTaskByContext(
	c echo.Context,
) (m models.Task, err error) {
	id, err := strconv.ParseInt(c.Param("TaskID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return
	}
	m, err = v.core.Tasks.Get(int(id))
	return
}

type Task struct {
	ID         int64              `json:"id"`
	ActionID   int                `json:"action_id,omitempty"`
	PlannerID  int                `json:"planner_id,omitempty"`
	CreateTime int64              `json:"create_time"`
	UpdateTime int64              `json:"update_time"`
	Status     models.TaskStatus  `json:"status"`
	Options    models.TaskOptions `json:"options"`
}

func makeTask(task models.Task) Task {
	resp := Task{
		ID:         task.ID,
		ActionID:   task.ActionID,
		PlannerID:  int(task.PlannerID),
		Status:     task.Status,
		Options:    task.Options,
		CreateTime: task.CreateTime,
		UpdateTime: task.UpdateTime,
	}
	switch task.Status {
	case models.StartingTask, models.RunningTask, models.AbortingTask:
		resp.UpdateTime = time.Now().Unix()
	}
	return resp
}

type TasksResponse struct {
	Tasks     []Task `json:"tasks"`
	NextBegin int64  `json:"next_begin,omitempty"`
}

func (v *View) ObserveTask(c echo.Context) error {
	task, ok := c.Get(taskKey).(models.Task)
	if !ok {
		return fmt.Errorf("task not extracted")
	}
	return c.JSON(http.StatusOK, makeTask(task))
}

func (v *View) AbortTask(c echo.Context) error {
	task, ok := c.Get(taskKey).(models.Task)
	if !ok {
		return fmt.Errorf("task not extracted")
	}
	if task.Status != models.RunningTask {
		return c.JSON(http.StatusNotModified, makeTask(task))
	}
	task.Status = models.AbortingTask
	if err := v.core.Tasks.Update(&task); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusOK, makeTask(task))
}

func (v *View) observeTaskLogs(c echo.Context, kind models.TaskLogType) error {
	task, ok := c.Get(taskKey).(models.Task)
	if !ok {
		return fmt.Errorf("task not extracted")
	}
	logs, err := v.core.TaskLogs.FindByTask(task.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	var builder strings.Builder
	for _, log := range logs {
		if log.Type == kind {
			builder.WriteString(log.Lines)
		}
	}
	return c.String(http.StatusOK, builder.String())
}

func (v *View) ObserveTaskStdoutLogs(c echo.Context) error {
	return v.observeTaskLogs(c, models.StdoutTaskLog)
}

func (v *View) ObserveTaskStderrLogs(c echo.Context) error {
	return v.observeTaskLogs(c, models.StderrTaskLog)
}

func (v *View) ObserveTaskSystemLogs(c echo.Context) error {
	return v.observeTaskLogs(c, models.SystemTaskLog)
}

func (v *View) registerTasks(g *echo.Group) {
	g.GET(
		"/v0/tasks/:task", v.ObserveTask, v.sessionAuth, v.extractTask,
		v.requireNodePermission(models.ObserveTaskPermission),
	)
	g.POST(
		"/v0/tasks/:task/abort", v.AbortTask, v.sessionAuth, v.requireAuth, v.extractTask,
		v.requireNodePermission(models.AbortTaskPermission),
	)
	g.GET(
		"/v0/tasks/:task/logs/stdout", v.ObserveTaskStdoutLogs, v.sessionAuth, v.extractTask,
		v.requireNodePermission(models.ObserveTaskPermission),
	)
	g.GET(
		"/v0/tasks/:task/logs/stderr", v.ObserveTaskStderrLogs, v.sessionAuth, v.extractTask,
		v.requireNodePermission(models.ObserveTaskPermission),
	)
	g.GET(
		"/v0/tasks/:task/logs/system", v.ObserveTaskSystemLogs, v.sessionAuth, v.extractTask,
		v.requireNodePermission(models.ObserveTaskPermission),
	)
}

func (v *View) extractTask(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		id, err := strconv.ParseInt(c.Param("task"), 10, 64)
		if err != nil {
			c.Logger().Warn(err)
			return c.NoContent(http.StatusBadRequest)
		}
		task, err := v.core.Tasks.Get(int(id))
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		c.Set(taskKey, task)
		action, err := v.core.Actions.Get(task.ActionID)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return err
		}
		c.Set(actionKey, action)
		if task.PlannerID != 0 {
			planner, err := v.core.Planners.Get(int(task.PlannerID))
			if err != nil {
				if err == sql.ErrNoRows {
					return c.NoContent(http.StatusNotFound)
				}
				c.Logger().Error(err)
				return err
			}
			c.Set(plannerKey, planner)
		}
		if planner, ok := c.Get(plannerKey).(models.Planner); ok {
			if planner.DirID != 0 {
				node, err := v.core.Nodes.Get(int(planner.DirID))
				if err != nil {
					if err == sql.ErrNoRows {
						return c.NoContent(http.StatusNotFound)
					}
					c.Logger().Error(err)
					return err
				}
				c.Set(nodeKey, node)
			}
		} else if action.DirID != 0 {
			node, err := v.core.Nodes.Get(int(action.DirID))
			if err != nil {
				if err == sql.ErrNoRows {
					return c.NoContent(http.StatusNotFound)
				}
				c.Logger().Error(err)
				return err
			}
			c.Set(nodeKey, node)
		}
		return next(c)
	}
}
