package api

import (
	"database/sql"
	"fmt"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
)

type TreeList struct {
	models.Node
	Dirs      []models.Node     `json:""`
	Actions   []models.Action   `json:""`
	Planners  []models.Planner  `json:""`
	Configs   []models.Config   `json:""`
	Resources []models.Resource `json:""`
	Secrets   []models.Secret   `json:""`
}

func (v *View) GetTreeList(c echo.Context) error {
	node, ok := c.Get(nodeKey).(models.Node)
	if !ok {
		return fmt.Errorf("node not fetched")
	}
	permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
	if !ok {
		return fmt.Errorf("permissions not fetched")
	}
	dirs, err := v.core.Nodes.FindByDir(node.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if dirs == nil || !permissions.HasPermission(models.ObserveNodePermission) {
		dirs = make([]models.Node, 0)
	}
	actions, err := v.core.Actions.FindByDir(node.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if actions == nil || !permissions.HasPermission(models.ObserveActionPermission) {
		actions = make([]models.Action, 0)
	}
	planners, err := v.core.Planners.FindByDir(node.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if planners == nil || !permissions.HasPermission(models.ObservePlannerPermission) {
		planners = make([]models.Planner, 0)
	}
	var resources []models.Resource
	if err := v.core.WithRoTx(
		c.Request().Context(),
		func(tx *sql.Tx) error {
			resources, err = v.core.Resources.FindByDirTx(tx, node.ID)
			return err
		},
	); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if resources == nil {
		resources = make([]models.Resource, 0)
	}
	configs, err := v.core.Configs.FindByDir(node.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if configs == nil {
		configs = make([]models.Config, 0)
	}
	secrets, err := v.core.Secrets.FindByDir(node.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.JSON(http.StatusOK, TreeList{
		Node:      node,
		Dirs:      dirs,
		Actions:   actions,
		Planners:  planners,
		Configs:   configs,
		Resources: resources,
		Secrets:   secrets,
	})
}

func (v *View) extractNode(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		id, err := strconv.Atoi(c.Param("node"))
		if err != nil {
			c.Logger().Warn(err)
			return c.JSON(http.StatusBadRequest, err)
		}
		node, err := v.core.Nodes.Get(id)
		if err != nil {
			if err == sql.ErrNoRows {
				return c.JSON(http.StatusNotFound, ErrorResponse{
					Message: fmt.Sprintf("node %d not found", id),
				})
			}
			c.Logger().Error(err)
			return err
		}
		c.Set(nodeKey, node)
		return next(c)
	}
}
