package api

import (
	"fmt"
	"math"
	"net/http"
	"strconv"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/library/go/yandex/blackbox"
)

type View struct {
	core *core.Core
}

func NewView(c *core.Core) *View {
	return &View{core: c}
}

// Ping returns "pong" string.
func (v *View) Ping(c echo.Context) error {
	return c.String(http.StatusOK, "pong")
}

// Health checks health of runner.
func (v *View) Health(c echo.Context) error {
	if err := v.core.DB.Ping(); err != nil {
		c.Logger().Error(err)
		return c.String(http.StatusInternalServerError, "unhealthy")
	}
	return c.String(http.StatusOK, "healthy")
}

const authTaskKey = "AuthTask"

// RegisterTask registers api for task.
func (v *View) RegisterTask(task models.Task, g *echo.Group) {
	// Register middleware
	g.Use(func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {
			c.Set(authTaskKey, task)
			return next(c)
		}
	})
	// Register handlers
	g.GET("/ping", v.Ping)
	v.registerResources(g)
}

// Register all handlers in echo group
func (v *View) Register(g *echo.Group) {
	// Service handlers.
	g.GET("/ping", v.Ping)
	g.GET("/health", v.Health)
	// Tree handlers.
	g.GET(
		"/tree/list/:node", v.GetTreeList,
		v.sessionAuth, v.extractNode,
		v.requireNodePermission(models.LegacyAuthPermission),
	)
	// Search.
	g.GET("/v0/search", v.Search, v.sessionAuth, v.requirePermission(models.LegacyAuthPermission))
	// Tasks handlers.
	g.GET("/tasks", v.GetTasks, v.sessionAuth, v.requirePermission(models.LegacyAuthPermission))
	g.GET("/tasks/:TaskID", v.GetTask, v.sessionAuth, v.requirePermission(models.LegacyAuthPermission))
	g.DELETE("/tasks/:TaskID", v.AbortTaskOld, v.sessionAuth, v.requirePermission(models.LegacyAuthPermission))
	g.GET("/tasks/:TaskID/logs/system", v.GetTaskSystemLogs, v.sessionAuth, v.requirePermission(models.LegacyAuthPermission))
	g.GET("/tasks/:TaskID/logs/stdout", v.GetTaskStdoutLogs, v.sessionAuth, v.requirePermission(models.LegacyAuthPermission))
	g.GET("/tasks/:TaskID/logs/stderr", v.GetTaskStderrLogs, v.sessionAuth, v.requirePermission(models.LegacyAuthPermission))
	v.registerRoles(g)
	v.registerHosts(g)
	v.registerNodes(g)
	v.registerActions(g)
	v.registerPlanners(g)
	v.registerTasks(g)
	v.registerConfigs(g)
	v.registerSecrets(g)
	v.registerResources(g)
}

func (v *View) RegisterSocket(g *echo.Group) {
	g.GET("/ping", v.Ping)
	g.GET("/health", v.Health)
	v.registerSocketRoles(g)
}

const (
	authBBSessionKey       = "auth_bb_session"
	authAccountKey         = "auth_account"
	authPermissionsKey     = "auth_permissions"
	authNodePermissionsKey = "auth_node_permissions"
	nodeKey                = "node"
	roleKey                = "role"
	accountRoleKey         = "account_role"
	accountNodeRoleKey     = "account_node_role"
	accountKey             = "account"
	actionKey              = "action"
	configKey              = "config"
	secretKey              = "secret"
	plannerKey             = "planner"
	resourceKey            = "resource"
	taskKey                = "task"
)

func getClientIP(c echo.Context) string {
	if ip := c.Request().Header.Get("X-Forwarded-For-Y"); ip != "" {
		return ip
	}
	return c.RealIP()
}

func getHost(host string) string {
	if len(host) > 0 && host[len(host)-1] != ']' {
		if pos := strings.LastIndex(host, ":"); pos != -1 {
			return host[:pos]
		}
	}
	return host
}

// extractBlackboxAuth tries to extract blackbox auth.
func (v *View) extractBlackboxAuth(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		if cfg := v.core.Config.FakeBBAuth; cfg != nil {
			session := blackbox.SessionIDResponse{
				User: blackbox.User{
					Login: cfg.YandexLogin,
					ID:    cfg.PassportUID,
				},
			}
			c.Set(authBBSessionKey, session)
			return next(c)
		}
		if _, ok := c.Get(authBBSessionKey).(blackbox.SessionIDResponse); ok {
			return next(c)
		}
		sessionID, err := c.Cookie("Session_id")
		if err != nil {
			c.Logger().Warn(err)
			return next(c)
		}
		session, err := v.core.Blackbox.SessionID(
			c.Request().Context(),
			blackbox.SessionIDRequest{
				SessionID:     sessionID.Value,
				UserIP:        getClientIP(c),
				Host:          getHost(c.Request().Host),
				GetUserTicket: false,
			},
		)
		if err != nil {
			c.Logger().Warn(err)
			return next(c)
		}
		c.Set(authBBSessionKey, *session)
		return next(c)
	}
}

// sessionAuth tries to authorize account using blackbox session.
func (v *View) sessionAuth(next echo.HandlerFunc) echo.HandlerFunc {
	nextWrap := func(c echo.Context) error {
		if _, ok := c.Get(authAccountKey).(models.Account); ok {
			return next(c)
		}
		session, ok := c.Get(authBBSessionKey).(blackbox.SessionIDResponse)
		if !ok {
			return next(c)
		}
		if session.User.ID >= math.MaxInt64 {
			return fmt.Errorf("passport UID is too large")
		}
		account, err := v.core.Accounts.GetByPassportUID(int64(session.User.ID))
		if err != nil {
			return err
		}
		// If login differs we should update login.
		if len(session.User.Login) > 0 && account.Login != session.User.Login {
			if err := v.core.Accounts.SyncTx(v.core.DB); err != nil {
				return err
			}
			account, err = v.core.Accounts.Get(account.ID)
			if err != nil {
				return err
			}
			if account.Login != session.User.Login {
				account.Login = session.User.Login
				if err := v.core.Accounts.UpdateTx(v.core.DB, account); err != nil {
					return err
				}
			}
		}
		c.Set(authAccountKey, account)
		return next(c)
	}
	return v.extractBlackboxAuth(nextWrap)
}

// requireAuth checks account authorization.
func (v *View) requireAuth(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		if _, ok := c.Get(authAccountKey).(models.Account); !ok {
			resp := ErrorResponse{Message: "auth required"}
			return c.JSON(http.StatusForbidden, resp)
		}
		return next(c)
	}
}

// extractPermissions extract permissions for user.
func (v *View) extractPermissions(
	next echo.HandlerFunc,
) echo.HandlerFunc {
	return func(c echo.Context) error {
		if _, ok := c.Get(authPermissionsKey).(core.PermissionSet); ok {
			return next(c)
		}
		permissions := core.PermissionSet{}
		if account, ok := c.Get(authAccountKey).(models.Account); ok {
			accountRoles, err := v.core.AccountRoles.FindByAccount(account.ID)
			if err != nil {
				c.Logger().Error("unable to find node roles")
				return fmt.Errorf("unable to find node roles")
			}
			for _, accountRole := range accountRoles {
				role, err := v.core.Roles.Get(accountRole.RoleID)
				if err != nil {
					c.Logger().Error(
						"unable to fetch role %d: %q",
						accountRole.RoleID, err,
					)
					continue
				}
				rolePermissions, err := role.GetPermissions()
				if err != nil {
					c.Logger().Error(
						"unable to fetch permissions for role %q: %q",
						role.Name, err,
					)
					continue
				}
				for _, permission := range rolePermissions {
					permissions[permission] = struct{}{}
				}
			}
		}
		c.Set(authPermissionsKey, permissions)
		return next(c)
	}
}

func (v *View) extendNodePermissions(
	c echo.Context, permissions core.PermissionSet, node models.Node,
) core.PermissionSet {
	nodePermissions := permissions.Clone()
	nodeIDs := map[int]struct{}{}
	for i, it := 0, node; node.ID != 0; i++ {
		if i > 100 {
			c.Logger().Error("too deep node tree")
			break
		}
		nodeIDs[it.ID] = struct{}{}
		if !it.InheritRoles || it.NodeID == 0 {
			break
		}
		jt, err := v.core.Nodes.Get(int(it.NodeID))
		if err != nil {
			c.Logger().Errorf(
				"unable to fetch node %d: %q", it.NodeID, err,
			)
			break
		}
		it = jt
	}
	if account, ok := c.Get(authAccountKey).(models.Account); ok {
		nodeRoles, err := v.core.AccountNodeRoles.FindByAccountNodes(
			account.ID, nodeIDs,
		)
		if err != nil {
			c.Logger().Error("unable to find account node roles")
			return nodePermissions
		}
		for _, nodeRole := range nodeRoles {
			role, err := v.core.Roles.Get(nodeRole.RoleID)
			if err != nil {
				c.Logger().Error(
					"unable to fetch role %d: %q",
					nodeRole.RoleID, err,
				)
				continue
			}
			rolePermissions, err := role.GetPermissions()
			if err != nil {
				c.Logger().Error(
					"unable to fetch permissions for role %q: %q",
					role.Name, err,
				)
				continue
			}
			for _, permission := range rolePermissions {
				nodePermissions[permission] = struct{}{}
			}
		}
	}
	return nodePermissions
}

func (v *View) extractNodePermissions(
	next echo.HandlerFunc,
) echo.HandlerFunc {
	nextWrap := func(c echo.Context) error {
		if _, ok := c.Get(authNodePermissionsKey).(core.PermissionSet); ok {
			return next(c)
		}
		node, ok := c.Get(nodeKey).(models.Node)
		if !ok {
			c.Logger().Error("node not extracted")
			return fmt.Errorf("node not extracted")
		}
		permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
		if !ok {
			c.Logger().Error("permissions not extracted")
			return fmt.Errorf("permissions not extracted")
		}
		c.Set(
			authNodePermissionsKey,
			v.extendNodePermissions(c, permissions, node),
		)
		return next(c)
	}
	return v.extractPermissions(nextWrap)
}

// requirePermission check that account has required permissions.
func (v *View) requirePermission(names ...string) echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		nextWrap := func(c echo.Context) error {
			resp := ErrorResponse{Message: "account missing permissions"}
			permissions, ok := c.Get(authPermissionsKey).(core.PermissionSet)
			if !ok {
				resp.MissingPermissions = names
				return c.JSON(http.StatusForbidden, resp)
			}
			for _, code := range names {
				if !permissions.HasPermission(code) {
					resp.MissingPermissions = append(resp.MissingPermissions, code)
				}
			}
			if len(resp.MissingPermissions) > 0 {
				return c.JSON(http.StatusForbidden, resp)
			}
			return next(c)
		}
		return v.extractPermissions(nextWrap)
	}
}

// requireNodePermission check that account has required permissions.
func (v *View) requireNodePermission(names ...string) echo.MiddlewareFunc {
	return func(next echo.HandlerFunc) echo.HandlerFunc {
		nextWrap := func(c echo.Context) error {
			resp := ErrorResponse{Message: "account missing permissions"}
			permissions, ok := c.Get(authNodePermissionsKey).(core.PermissionSet)
			if !ok {
				resp.MissingPermissions = names
				return c.JSON(http.StatusForbidden, resp)
			}
			for _, code := range names {
				if !permissions.HasPermission(code) {
					resp.MissingPermissions = append(resp.MissingPermissions, code)
				}
			}
			if len(resp.MissingPermissions) > 0 {
				return c.JSON(http.StatusForbidden, resp)
			}
			return next(c)
		}
		return v.extractNodePermissions(nextWrap)
	}
}

func getEventOptions(c echo.Context) []models.EventOption {
	var options []models.EventOption
	if account, ok := c.Get(authAccountKey).(models.Account); ok {
		options = append(options, models.WithUser(account.ID))
	}
	if task, ok := c.Get(authTaskKey).(models.Task); ok {
		options = append(options, models.WithTask(task.ID))
	}
	return options
}

// Try to read int from query param or return value
func getIntQueryParam(c echo.Context, name string, value int) int {
	str := c.QueryParam(name)
	if str == "" {
		return value
	}
	val, err := strconv.Atoi(str)
	if err != nil {
		return value
	}
	return val
}

type InvalidField struct {
	Message string `json:"message"`
}

type InvalidFields map[string]InvalidField

// ErrorResponse represents response with information about error.
type ErrorResponse struct {
	// Message contains common message for response.
	Message string `json:"message"`
	// InvalidFields contains information about invalid fields.
	InvalidFields InvalidFields `json:"invalid_fields,omitempty"`
	// MissingPermissions contains list of required permissions.
	MissingPermissions []string `json:"missing_permissions,omitempty"`
	// FieldErrors contains errors for fields.
	FieldErrors models.FieldListError `json:",omitempty"`
}
