package main

import (
	"context"
	"fmt"
	"math/rand"
	"net"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/drive/library/go/echolog"
	"a.yandex-team.ru/drive/library/go/gocfg"
	"a.yandex-team.ru/drive/runner/api"
	"a.yandex-team.ru/drive/runner/config"
	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/worker"
	"a.yandex-team.ru/library/go/core/log"

	_ "github.com/ClickHouse/clickhouse-go"
	_ "github.com/jackc/pgx/v4/stdlib"
	_ "github.com/mattn/go-sqlite3"
)

func serverMain(cmd *cobra.Command, _ []string) {
	rand.Seed(time.Now().UnixNano())
	configPath, err := cmd.Flags().GetString("config")
	if err != nil {
		panic(err)
	}
	cfg := config.Config{
		SocketFile: "/tmp/runner-server.sock",
	}
	if err := gocfg.ParseFile(configPath, &cfg); err != nil {
		panic(err)
	}
	if cfg.Server == nil && cfg.Worker == nil {
		panic(fmt.Errorf("server or worker section should be specified"))
	}
	c, err := core.NewCore(&cfg)
	if err != nil {
		panic(err)
	}
	logger := c.Logger("")
	if err := c.Start(); err != nil {
		logger.Fatal("Unable to start core", log.Error(err))
	}
	defer c.Stop()
	quit := make(chan os.Signal, 1)
	signal.Notify(quit, os.Interrupt, syscall.SIGTERM)
	if cfg.Worker != nil {
		w, err := worker.NewWorker(c)
		if err != nil {
			logger.Fatal("Unable to start worker", log.Error(err))
		}
		w.Start()
	}
	isServerError := func(err error) bool {
		return err != nil && err != http.ErrServerClosed
	}
	if cfg.Server != nil {
		s, err := api.NewServer(c)
		if err != nil {
			logger.Fatal("Unable to start server", log.Error(err))
		}
		timeout := cfg.Server.ShutdownTimeout
		if timeout == 0 {
			logger.Warn("Server does not have shutdown_timeout")
			timeout = time.Second * 20
		}
		defer func() {
			ctx, cancel := context.WithTimeout(context.Background(), 20*time.Second)
			defer cancel()
			if err := s.Shutdown(ctx); err != nil {
				logger.Error("Unable to shutdown server", log.Error(err))
			}
		}()
		go func() {
			if err := s.Start(cfg.Server.Addr); isServerError(err) {
				logger.Error("HTTP server error", log.Error(err))
			}
			signal.Stop(quit)
		}()
	}
	if file := cfg.SocketFile; file != "" {
		if err := os.Remove(file); err != nil && !os.IsNotExist(err) {
			logger.Fatal("Unable to remove socket file", log.Error(err))
		}
		sock := echo.New()
		if sock.Listener, err = net.Listen("unix", file); err != nil {
			logger.Fatal("Unable to open unix file", log.Error(err))
		}
		sock.HideBanner, sock.HidePort = true, true
		sock.Use(
			echolog.Logger(c.Logger("socket")),
			middleware.Recover(),
			middleware.Gzip(),
		)
		sock.Pre(middleware.RemoveTrailingSlash())
		api.NewView(c).RegisterSocket(sock.Group("/socket"))
		defer func() {
			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()
			if err := sock.Shutdown(ctx); err != nil {
				logger.Error("Unable to shutdown server", log.Error(err))
			}
		}()
		go func() {
			if err := sock.Start(""); isServerError(err) {
				logger.Error("HTTP server error", log.Error(err))
			}
			signal.Stop(quit)
		}()
	}
	<-quit
}

func main() {
	cmd := cobra.Command{}
	cmd.PersistentFlags().String("config", "config.json", "")
	serverCmd := cobra.Command{
		Use: "server",
		Run: serverMain,
	}
	cmd.AddCommand(&serverCmd)
	if err := cmd.Execute(); err != nil {
		panic(err)
	}
}
