package config

import (
	"time"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/library/go/secret"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/zootopia/analytics/drive/config"
)

// Config contains configuration for Drive Runner.
type Config struct {
	// DB contains DB configuration.
	DB gosql.Config `json:"db"`
	// TVM contains tvm configuration.
	TVM *TVM `json:"tvm"`
	// HostName contains host name.
	HostName secret.Secret `json:"host_name"`
	// HostLabels contains host labels.
	HostLabels map[string]string `json:"host_labels"`
	// Server contains server configuration.
	Server *Server `json:"server"`
	// Worker contains worker configuration.
	Worker *Worker `json:"worker"`
	// SocketFile contains socket file configuration.
	SocketFile string `json:"socket_file"`
	// LogLevel contains log level configuration.
	LogLevel log.Level `json:"log_level"`
	// Solomon contains Solomon configuration.
	Solomon *Solomon `json:"solomon"`
	// FakeBBAuth contains fake blackbox session.
	FakeBBAuth *FakeBBAuth `json:"fake_bb_auth"`
	// Deprecated.
	Domain    string                   `json:""`
	Secrets   map[string]secret.Secret `json:""`
	RootDir   int                      `json:""`
	Arcadia   *HTTPOAuthFileProvider   `json:""`
	Sandbox   *HTTPOAuthFileProvider   `json:""`
	Mail      Mail                     `json:""`
	SystemDir string                   `json:""`
	MDS       MDS                      `json:""`
	Yav       Yav                      `json:""`
	YT        config.YT                `json:""`
}

// Server represents HTTP server configuration.
type Server struct {
	// RootName contains root name.
	RootName string `json:"root_name"`
	// Addr contains server address.
	Addr string `json:"addr"`
	// ShutdownTimeout contains server shutdown timeout.
	ShutdownTimeout time.Duration `json:"shutdown_timeout"`
	// StaticDir contains path to static dir.
	StaticDir string `json:"static_dir"`
	// OldStaticDir contains path to legacy static dir.
	OldStaticDir string `json:"old_static_dir"`
	// OldTemplatesDir contains path to legacy templates dir.
	OldTemplatesDir string `json:"old_templates_dir"`
}

// Worker represents worker configuration.
type Worker struct {
	// Arcadia contains arcadia configuration.
	Arcadia *HTTPOAuthFileProvider `json:"arcadia"`
	// Sandbox contains sandbox configuration.
	Sandbox *HTTPOAuthFileProvider `json:"sandbox"`
	// CacheDir contains cache directory.
	CacheDir string `json:"cache_dir"`
	// PortoMaxConnections contains maximum amount of connections to porto.
	PortoMaxConnections int `json:"porto_max_connections"`
}

// TVM represents tvm configuration.
type TVM struct {
	// Source contains source.
	Source tvm.ClientID `json:"source"`
	// Secret contains secret.
	Secret secret.Secret `json:"secret"`
	// Targets contains targets.
	Targets map[string]tvm.ClientID `json:"targets"`
	// CacheDir contains cache directory.
	CacheDir string `json:"cache_dir"`
	// BBEnv contains Blackbox environment name.
	BBEnv string `json:"bb_env"`
}

// Solomon contains configuration for solomon.
type Solomon struct {
	// Endpoint.
	Endpoint string `json:"endpoint"`
	// Project.
	Project string `json:"project"`
	// Cluster.
	Cluster string `json:"cluster"`
	// Service.
	Service string `json:"service"`
	// Target represents name of TVM target.
	Target string `json:"target"`
}

type FakeBBAuth struct {
	YandexLogin string `json:"yandex_login"`
	PassportUID uint64 `json:"passport_uid"`
}

type MDS struct {
	BaseURL   string        `json:""`
	Bucket    string        `json:""`
	AccessKey secret.Secret `json:""`
	Secret    secret.Secret `json:""`
	Prefix    string        `json:""`
	Key       secret.Secret `json:""`
}

type Mail struct {
	Host string `json:""`
	Port int    `json:""`
	From string `json:""`
}

// Yav represents yav.yandex-team.ru endpoint.
type Yav struct {
	Endpoint string        `json:""`
	Token    secret.Secret `json:""`
}

// HTTPOAuthFileProvider represents file provider config that uses OAuth.
type HTTPOAuthFileProvider struct {
	// BaseURL contains URL of file provider.
	BaseURL string `json:"base_url"`
	// Token contains OAuth token.
	Token secret.Secret `json:"token"`
}
