package core

import (
	"context"
	"database/sql"
	"fmt"
	"os"
	"runtime"
	"sync"
	"time"

	"a.yandex-team.ru/drive/library/go/auth"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/library/go/secret"
	"a.yandex-team.ru/drive/library/go/solomon"
	"a.yandex-team.ru/drive/runner/config"
	"a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	tp2 "a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
)

type PermissionSet map[string]struct{}

// HasPermission return that permission set has specified permission.
func (s PermissionSet) HasPermission(code string) bool {
	_, ok := s[code]
	return ok
}

func (s PermissionSet) Clone() PermissionSet {
	clone := PermissionSet{}
	for key := range s {
		clone[key] = struct{}{}
	}
	return clone
}

// Core represents core for DAJR.
type Core struct {
	Config   *config.Config
	DB       *gosql.DB
	Locks    *models.LockStore
	Hosts    *models.HostStore
	Nodes    *models.NodeStore
	Actions  *models.ActionStore
	Planners *models.PlannerStore
	// PlannerStates contains store for planner states.
	PlannerStates *models.PlannerStateStore
	// Tasks contains store for tasks.
	Tasks     *models.TaskStore
	TaskLogs  *models.TaskLogStore
	Secrets   *models.SecretStore
	Configs   *models.ConfigStore
	Resources *models.ResourceStore
	// Roles contains store for roles.
	Roles *models.RoleStore
	// Accounts contains store for users.
	Accounts *models.AccountStore
	// AccountRoles contains store for user roles.
	AccountRoles *models.AccountRoleStore
	// AccountNodeRoles contains store for node roles.
	AccountNodeRoles *models.AccountNodeRoleStore
	// TVM contains TVM2 client.
	TVM tvm.Client
	// Solomon contains client for solomon.
	Solomon *solomon.Client
	// Blackbox contains Blackbox client.
	Blackbox blackbox.Client
	// logger contains logger.
	logger log.Logger
	// host contains current host.
	host models.Host
	// mutex contains mutex for core mutations.
	mutex sync.RWMutex
	// hostContext contains core context for host.
	hostContext context.Context
	// hostCancel represents function that can cancel core host context.
	hostCancel context.CancelFunc
	// hostWaiter represents core waiter for host ping loop.
	hostWaiter sync.WaitGroup
	// taskContext contains core context.
	taskContext context.Context
	// taskCancel represents function that can cancel core context.
	taskCancel context.CancelFunc
	// taskWaiter represents core waiter for all tasks.
	taskWaiter sync.WaitGroup
	// runningTasks contains list of all running tasks.
	runningTasks sync.Map
}

// NewCore creates a new instance of Core.
func NewCore(cfg *config.Config) (*Core, error) {
	db, err := gosql.NewDB(cfg.DB)
	if err != nil {
		return nil, err
	}
	logger, err := zap.NewDeployLogger(cfg.LogLevel)
	if err != nil {
		return nil, err
	}
	secrets, err := models.NewSecretStore(
		db, "secret", "secret_event", cfg.Yav, logger,
	)
	if err != nil {
		return nil, err
	}
	c := Core{
		Config:   cfg,
		DB:       db,
		Locks:    models.NewLockStore(db, "lock", logger),
		Hosts:    models.NewHostStore(db, "host"),
		Nodes:    models.NewNodeStore(db, "node", "node_event"),
		Roles:    models.NewRoleStore(db, "role", "role_event"),
		Accounts: models.NewAccountStore(db, "account", "account_event"),
		AccountRoles: models.NewAccountRoleStore(
			db, "account_role", "account_role_event",
		),
		AccountNodeRoles: models.NewAccountNodeRoleStore(
			db, "account_node_role", "account_node_role_event",
		),
		Planners:      models.NewPlannerStore(db, "planner", "planner_event"),
		PlannerStates: models.NewPlannerStateStore(db, "planner_state"),
		Actions:       models.NewActionStore(db, "action", "action_event"),
		Configs:       models.NewConfigStore(db, "config", "config_event"),
		Secrets:       secrets,
		Resources: models.NewResourceStore(
			db, "resource", "resource_event", cfg.MDS,
		),
		Tasks:        models.NewTaskStore(db, "task"),
		TaskLogs:     models.NewTaskLogStore(db, "task_log"),
		logger:       logger,
		runningTasks: sync.Map{},
	}
	if tvmCfg := cfg.TVM; tvmCfg != nil {
		settings := tp2.TvmAPISettings{
			SelfID:       tvmCfg.Source,
			DiskCacheDir: tvmCfg.CacheDir,
			ServiceTicketOptions: tp2.NewAliasesOptions(
				tvmCfg.Secret.Secret(), tvmCfg.Targets,
			),
			EnableServiceTicketChecking: true,
		}
		tvm2, err := tp2.NewAPIClient(settings, logger)
		if err != nil {
			return nil, err
		}
		c.TVM = tvm2
		bbEnv, err := getBlackboxEnv(cfg.TVM.BBEnv)
		if err != nil {
			return nil, err
		}
		bb, err := httpbb.NewClient(bbEnv, httpbb.WithTVM(tvm2))
		if err != nil {
			return nil, err
		}
		c.Blackbox = bb
		if cfg := cfg.Solomon; cfg != nil {
			tvmAuth := auth.ServiceTicket{TVM: tvm2, Target: cfg.Target}
			hostname, err := getHostName(c.Config.HostName)
			if err != nil {
				return nil, err
			}
			c.Solomon, err = solomon.NewClient(
				cfg.Project, cfg.Cluster, cfg.Service,
				solomon.WithAuth(tvmAuth),
				solomon.WithTags(map[string]string{"host": hostname}),
			)
			if err != nil {
				return nil, err
			}
		}
	}
	return &c, nil
}

// Logger returns logger.
func (c *Core) Logger(name string) log.Logger {
	if name == "" {
		return c.logger
	}
	return c.logger.WithName(name)
}

// startManagers should call start function for
// each manager that should be started.
func (c *Core) startManagers(
	start func(string, models.Manager, time.Duration),
) {
	start("nodes", c.Nodes, time.Second)
	start("actions", c.Actions, time.Second)
	start("planners", c.Planners, time.Second)
	start("secrets", c.Secrets, time.Second)
	start("configs", c.Configs, time.Second)
	start("accounts", c.Accounts, time.Second)
	start("roles", c.Roles, time.Second)
	start("account_roles", c.AccountRoles, time.Second)
	start("account_node_roles", c.AccountNodeRoles, time.Second)
	// The following stores does not require start:
	//   - ResourceManager.
}

func (c *Core) startManagerLoops() error {
	// Start all managers and wait for their init.
	errs := make(chan error)
	count := 0
	c.startManagers(func(n string, m models.Manager, d time.Duration) {
		count++
		c.hostWaiter.Add(1)
		go c.runManagerLoop(n, m, d, errs)
	})
	var err error
	for i := 0; i < count; i++ {
		if errLast := <-errs; errLast != nil {
			err = errLast
		}
	}
	return err
}

func (c *Core) startHost() error {
	name, err := getHostName(c.Config.HostName)
	if err != nil {
		return fmt.Errorf("unable to get hostname: %s", err)
	}
	if err := gosql.WithTx(c.DB, func(tx *sql.Tx) error {
		c.host, err = c.Hosts.GetByNameTx(tx, name)
		if err == sql.ErrNoRows {
			c.host, err = c.Hosts.CreateTx(tx, models.Host{
				Name: name,
				Config: models.HostConfig{
					Active:       true,
					EnableLeader: true,
				},
			})
		}
		return err
	}); err != nil {
		return fmt.Errorf("unable to choose host: %s", err)
	}
	c.hostWaiter.Add(2)
	go c.hostPingLoop()
	go c.hostStatLoop()
	return nil
}

// TODO(iudovin@): Replace old solomon signals.
func (c *Core) hostPingLoop() {
	defer c.hostWaiter.Done()
	okSignal := c.Signal("host_ping.ok_sum", nil)
	errorSignal := c.Signal("host_ping.error_sum", nil)
	ping := func() {
		if err := c.hostPing(); err != nil {
			c.logger.Error(
				"Unable to ping host",
				log.Int("host_id", c.Host().ID),
				log.Error(err),
			)
			errorSignal.Add(1)
		} else {
			c.logger.Debug(
				"Host ping successful",
				log.Int("host_id", c.Host().ID),
			)
			okSignal.Add(1)
		}
	}
	// Setup ping timeout.
	ticker := time.NewTicker(time.Second)
	defer ticker.Stop()
	for {
		select {
		case <-c.hostContext.Done():
			ping()
			return
		case <-ticker.C:
			ping()
		}
	}
}

func (c *Core) hostStatLoop() {
	defer c.hostWaiter.Done()
	// Setup metrics timeout.
	ticker := time.NewTicker(5 * time.Second)
	defer ticker.Stop()
	goroutinesSignal := c.Signal("stats.goroutines_last", nil)
	heapAllocSignal := c.Signal("stats.heap_alloc_last", nil)
	heapSysSignal := c.Signal("stats.heap_sys_last", nil)
	stackSysSignal := c.Signal("stats.stack_sys_last", nil)
	dbConnsOpenSignal := c.Signal("stats.db_conns_open_last", nil)
	dbConnsInUseSignal := c.Signal("stats.db_conns_in_use_last", nil)
	dbConnsIdleSignal := c.Signal("stats.db_conns_idle_last", nil)
	for {
		select {
		case <-c.hostContext.Done():
			return
		case <-ticker.C:
			goroutines := runtime.NumGoroutine()
			goroutinesSignal.Set(float64(goroutines))
			var mem runtime.MemStats
			runtime.ReadMemStats(&mem)
			heapAllocSignal.Set(float64(mem.HeapAlloc))
			heapSysSignal.Set(float64(mem.HeapSys))
			stackSysSignal.Set(float64(mem.StackSys))
			stats := c.DB.Stats()
			dbConnsOpenSignal.Set(float64(stats.OpenConnections))
			dbConnsInUseSignal.Set(float64(stats.InUse))
			dbConnsIdleSignal.Set(float64(stats.Idle))
		}
	}
}

func (c *Core) hostPing() error {
	c.mutex.Lock()
	defer c.mutex.Unlock()
	c.host.State.RunningTasks = c.GetRunningTasks()
	c.host.State.Labels = c.Config.HostLabels
	if err := c.Hosts.Ping(&c.host); err != nil {
		return err
	}
	return c.Hosts.Reload(&c.host)
}

// Host returns current host.
func (c *Core) Host() models.Host {
	c.mutex.RLock()
	defer c.mutex.RUnlock()
	return c.host
}

// Start starts core.
func (c *Core) Start() error {
	if c.hostCancel != nil {
		return fmt.Errorf("core already started")
	}
	c.hostContext, c.hostCancel = context.WithCancel(context.Background())
	c.taskContext, c.taskCancel = context.WithCancel(c.hostContext)
	if err := c.startManagerLoops(); err != nil {
		c.Stop()
		return err
	}
	if err := c.startHost(); err != nil {
		c.Stop()
		return err
	}
	return nil
}

// Stop stops all managers and waits for exit.
func (c *Core) Stop() {
	if c.hostCancel == nil {
		return
	}
	c.taskCancel()
	c.taskWaiter.Wait()
	c.hostCancel()
	c.hostWaiter.Wait()
	if c.Solomon != nil {
		c.Solomon.Close()
	}
	c.hostCancel, c.taskCancel = nil, nil
}

// StartTask starts function in new goroutine.
//
// Context passed to function will be cancelled when core is stopped.
func (c *Core) StartTask(name string, task func(ctx context.Context)) {
	if _, ok := c.runningTasks.LoadOrStore(name, struct{}{}); ok {
		panic(fmt.Errorf("task %q already running", name))
	}
	c.startTask(func(ctx context.Context) {
		defer c.runningTasks.Delete(name)
		task(ctx)
	})
}

// StartDaemon starts function in new goroutine.
//
// Context passed to function will be cancelled when core is stopped.
func (c *Core) StartDaemon(
	name string, task func(ctx context.Context) error,
) {
	runTask := func(ctx context.Context) error {
		if _, ok := c.runningTasks.LoadOrStore(name, struct{}{}); ok {
			return fmt.Errorf("task %q already running", name)
		}
		defer c.runningTasks.Delete(name)
		return task(ctx)
	}
	c.startTask(func(ctx context.Context) {
		ticker := time.NewTicker(5 * time.Second)
		defer ticker.Stop()
		if err := runTask(ctx); err != nil {
			c.logger.Error("Daemon task failed", log.Error(err))
		}
		for {
			select {
			case <-ctx.Done():
				return
			case <-ticker.C:
				// Kostyl for random select.
				select {
				case <-ctx.Done():
					return
				default:
				}
				if err := runTask(ctx); err != nil {
					c.logger.Error("Daemon task failed", log.Error(err))
				}
			}
		}
	})
}

// StartClusterDaemon starts infinite loop of task function calls.
//
// Its guaranteed that only one task in whole cluster will be running
// at the same time. Note, that task function should finish its work
// as soon as possible when context is cancelled.
func (c *Core) StartClusterDaemon(
	name string, task func(ctx context.Context) error,
) {
	runTask := func(ctx context.Context) error {
		if _, ok := c.runningTasks.LoadOrStore(name, struct{}{}); ok {
			return fmt.Errorf("task %q already running", name)
		}
		defer c.runningTasks.Delete(name)
		return task(ctx)
	}
	c.startTask(func(ctx context.Context) {
		ticker := time.NewTicker(5 * time.Second)
		defer ticker.Stop()
		if err := c.Locks.WithLock(
			ctx, name, c.Host().ID, runTask,
		); badError(err) {
			c.logger.Error(
				"Unable to acquire lock",
				log.Int("host_id", c.Host().ID),
				log.Error(err),
			)
		}
		for {
			select {
			case <-ctx.Done():
				return
			case <-ticker.C:
				// Kostyl for random select.
				select {
				case <-ctx.Done():
					return
				default:
				}
				if err := c.Locks.WithLock(
					ctx, name, c.Host().ID, runTask,
				); badError(err) {
					c.logger.Error(
						"Unable to acquire lock",
						log.Int("host_id", c.Host().ID),
						log.Error(err),
					)
				}
			}
		}
	})
}

func (c *Core) startTask(task func(ctx context.Context)) {
	c.taskWaiter.Add(1)
	go func() {
		defer c.taskWaiter.Done()
		task(c.taskContext)
	}()
}

func badError(err error) bool {
	return err != nil && err != models.ErrLockAcquired &&
		err != models.ErrLockReleased && err != context.Canceled
}

// GetRunningTasks returns currently running tasks by core.
func (c *Core) GetRunningTasks() []string {
	var tasks []string
	c.runningTasks.Range(func(key, value interface{}) bool {
		tasks = append(tasks, key.(string))
		return true
	})
	return tasks
}

// WithTx runs function with SQL transaction.
func (c *Core) WithTx(
	ctx context.Context, fn func(*sql.Tx) error,
) (err error) {
	return gosql.WithTxContext(ctx, c.DB, nil, fn)
}

// WithRoTx runs function with read-only SQL transaction.
func (c *Core) WithRoTx(
	ctx context.Context, fn func(*sql.Tx) error,
) (err error) {
	opts := &sql.TxOptions{ReadOnly: true}
	return gosql.WithTxContext(ctx, c.DB, opts, fn)
}

// Signal returns signal wrapper for solomon monitoring.
func (c *Core) Signal(name string, tags map[string]string) *solomon.Signal {
	if c.Solomon == nil {
		return solomon.StubSignal
	}
	return c.Solomon.Signal(name, tags)
}

var withReadOnly = gosql.WithTxOptions(&sql.TxOptions{ReadOnly: true})

func (c *Core) runManagerLoop(
	name string, manager models.Manager, delay time.Duration,
	errs chan<- error,
) {
	defer c.hostWaiter.Done()
	err := gosql.WithTx(
		c.DB,
		func(tx *sql.Tx) error {
			return manager.InitTx(tx)
		},
		gosql.WithContext(c.hostContext),
		withReadOnly,
	)
	errs <- err
	if err != nil {
		return
	}
	ticker := time.NewTicker(delay)
	defer ticker.Stop()
	okSignal := c.Signal(
		"store_cache_sync.ok_sum",
		map[string]string{"store": name},
	)
	errorSignal := c.Signal(
		"store_cache_sync.error_sum",
		map[string]string{"store": name},
	)
	for {
		select {
		case <-c.hostContext.Done():
			return
		case <-ticker.C:
			if err := gosql.WithTx(
				c.DB,
				func(tx *sql.Tx) error {
					return manager.SyncTx(tx)
				},
				gosql.WithContext(c.hostContext),
				withReadOnly,
			); err != nil {
				c.logger.Error(
					"Unable to sync manager",
					log.String("manager", name),
					log.Error(err),
				)
				errorSignal.Add(1)
			} else {
				okSignal.Add(1)
			}
		}
	}
}

func getBlackboxEnv(name string) (httpbb.Environment, error) {
	switch name {
	case "intranet", "":
		return httpbb.IntranetEnvironment, nil
	case "test":
		return httpbb.TestEnvironment, nil
	case "prod":
		return httpbb.ProdEnvironment, nil
	case "mimino":
		return httpbb.MiminoEnvironment, nil
	}
	return httpbb.Environment{}, fmt.Errorf(
		"unsupported Blackbox environment: %q", name,
	)
}

func getHostName(name secret.Secret) (string, error) {
	if n := name.Secret(); n != "" {
		return n, nil
	}
	return os.Hostname()
}
