package front

import (
	"database/sql"
	"fmt"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/models"
)

func (v *View) GetActionInfo(c echo.Context) error {
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	tasks, err := v.app.Tasks.FindByAction(int(action.ID))
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Render(http.StatusOK, "action.html", map[string]interface{}{
		"Action": action,
		"Tasks":  tasks,
	})
}

func (v *View) GetActionEdit(c echo.Context) error {
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Render(http.StatusOK, "action_edit.html", map[string]interface{}{
		"Action": action,
	})
}

func (v *View) PostActionEdit(c echo.Context) error {
	params, err := c.FormParams()
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	options, err := parseActionOptions(params)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	action.Options = options
	user := c.Get("auth_account").(models.Account)
	if err := v.ensureUserCreated(&user); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := v.app.WithTx(c.Request().Context(), func(tx *sql.Tx) error {
		return v.app.Actions.UpdateTx(tx, action, models.WithUser(user.ID))
	}); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusFound,
		fmt.Sprintf("/action/%d", action.ID),
	)
}

func (v *View) GetActionRun(c echo.Context) error {
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Render(http.StatusOK, "action_run.html", action)
}

func (v *View) GetActionPlan(c echo.Context) error {
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	user, ok := c.Get("auth_account").(models.Account)
	if !ok {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Render(http.StatusOK, "action_plan.html", map[string]interface{}{
		"Action": action, "User": user,
	})
}

func (v *View) GetActionTreeRoot(c echo.Context) error {
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if action.DirID == 0 {
		return c.NoContent(http.StatusNotFound)
	}
	c.SetParamNames("ActionID", "DirID")
	c.SetParamValues(
		c.Param("ActionID"),
		fmt.Sprintf("%d", action.DirID),
	)
	return v.GetActionTreeList(c)
}

func (v *View) GetActionTreeList(c echo.Context) error {
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if action.DirID == 0 {
		return c.NoContent(http.StatusNotFound)
	}
	dirID, err := strconv.ParseInt(c.Param("DirID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	dir, err := v.app.Nodes.Get(int(dirID))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	dirs, err := v.app.Nodes.FindByDir(int(dirID))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if dirs == nil {
		dirs = make([]models.Node, 0)
	}
	if dir.NodeID != 0 {
		parent, err := v.app.Nodes.Get(int(dir.NodeID))
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		return c.Render(http.StatusOK, "action_tree.html", map[string]interface{}{
			"Parent": parent, "Dirs": dirs, "Dir": dir, "Action": action,
		})
	}
	return c.Render(http.StatusOK, "action_tree.html", map[string]interface{}{
		"Dirs": dirs, "Dir": dir, "Action": action,
	})
}

func (v *View) PostActionTreeList(c echo.Context) error {
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	dirID, err := strconv.ParseInt(c.Param("DirID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	dir, err := v.app.Nodes.Get(int(dirID))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	action.DirID = models.NInt(dir.ID)
	user := c.Get("auth_account").(models.Account)
	if err := v.ensureUserCreated(&user); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := v.app.WithTx(c.Request().Context(), func(tx *sql.Tx) error {
		return v.app.Actions.UpdateTx(tx, action, models.WithUser(user.ID))
	}); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusFound,
		fmt.Sprintf("/action/%d", action.ID),
	)
}
