package front

import (
	"database/sql"
	"fmt"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/models"
)

func (v *View) GetConfigInfo(c echo.Context) error {
	config, err := v.getConfigByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Render(http.StatusOK, "config.html", map[string]interface{}{
		"Config": config,
	})
}

func (v *View) GetConfigEdit(c echo.Context) error {
	config, err := v.getConfigByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Render(http.StatusOK, "config_edit.html", map[string]interface{}{
		"Config": config,
	})
}

func (v *View) PostConfigEdit(c echo.Context) error {
	params, err := c.FormParams()
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	config, err := v.getConfigByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	config.Title = params.Get("Title")
	config.Description = params.Get("Description")
	config.Data = params.Get("Data")
	user := c.Get("auth_account").(models.Account)
	if err := v.ensureUserCreated(&user); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := v.app.WithTx(c.Request().Context(), func(tx *sql.Tx) error {
		return v.app.Configs.UpdateTx(tx, config, models.WithUser(user.ID))
	}); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusFound,
		fmt.Sprintf("/config/%d", config.ID),
	)
}

func (v *View) GetConfigTreeRoot(c echo.Context) error {
	config, err := v.getConfigByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if config.DirID == 0 {
		return c.NoContent(http.StatusNotFound)
	}
	c.SetParamNames("ConfigID", "DirID")
	c.SetParamValues(
		c.Param("ConfigID"),
		fmt.Sprintf("%d", config.DirID),
	)
	return v.GetConfigTreeList(c)
}

func (v *View) GetConfigTreeList(c echo.Context) error {
	config, err := v.getConfigByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if config.DirID == 0 {
		return c.NoContent(http.StatusNotFound)
	}
	dirID, err := strconv.ParseInt(c.Param("DirID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	dir, err := v.app.Nodes.Get(int(dirID))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	dirs, err := v.app.Nodes.FindByDir(int(dirID))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if dirs == nil {
		dirs = make([]models.Node, 0)
	}
	return c.Render(http.StatusOK, "config_tree.html", map[string]interface{}{
		"Dirs": dirs, "Dir": dir, "Config": config,
	})
}

func (v *View) PostConfigTreeList(c echo.Context) error {
	config, err := v.getConfigByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	dirID, err := strconv.ParseInt(c.Param("DirID"), 10, 64)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	dir, err := v.app.Nodes.Get(int(dirID))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	config.DirID = models.NInt(dir.ID)
	user := c.Get("auth_account").(models.Account)
	if err := v.ensureUserCreated(&user); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if err := v.app.WithTx(c.Request().Context(), func(tx *sql.Tx) error {
		return v.app.Configs.UpdateTx(tx, config, models.WithUser(user.ID))
	}); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusFound,
		fmt.Sprintf("/config/%d", config.ID),
	)
}
