package front

import (
	"encoding/json"
	"fmt"
	"html/template"
	"net/url"
	"path"
	"regexp"
	"strconv"
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/library/go/cron"
	"a.yandex-team.ru/drive/runner/config"
	"a.yandex-team.ru/drive/runner/models"
)

func isChecked(value string) bool {
	return value == "on"
}

func parsePlannerSettings(values url.Values) (models.PlannerSettings, error) {
	result := models.PlannerSettings{}
	schedule, ok := getParamValue(values, "Schedule")
	if ok && schedule != "" {
		scheduleVal, err := cron.Parse(schedule)
		if err == nil {
			result.Schedule = scheduleVal
		}
	}
	repeatDelay, ok := getParamValue(values, "RepeatDelay")
	if ok {
		repeatDelayInt, err := strconv.Atoi(repeatDelay)
		if err == nil {
			result.RepeatDelay = &repeatDelayInt
		}
	}
	failureDelay, ok := getParamValue(values, "FailureDelay")
	if ok {
		failureDelayInt, err := strconv.Atoi(failureDelay)
		if err == nil {
			result.FailureDelay = &failureDelayInt
		}
	}
	allowOverlaps, ok := getParamValue(values, "AllowOverlaps")
	if ok {
		result.AllowOverlaps = allowOverlaps == "true"
	}
	listenSuccess, ok := getParamValue(values, "ListenSuccess")
	if ok {
		ids := strings.Split(listenSuccess, ",")
		for _, strID := range ids {
			strID = strings.TrimSpace(strID)
			if len(strID) == 0 {
				continue
			}
			if id, err := strconv.Atoi(strID); err == nil {
				result.ListenSuccess = append(result.ListenSuccess, id)
			}
		}
	}
	listenFailure, ok := getParamValue(values, "ListenFailure")
	if ok {
		ids := strings.Split(listenFailure, ",")
		for _, strID := range ids {
			strID = strings.TrimSpace(strID)
			if len(strID) == 0 {
				continue
			}
			if id, err := strconv.Atoi(strID); err == nil {
				result.ListenFailure = append(result.ListenFailure, id)
			}
		}
	}
	failureMails, ok := getParamValue(values, "FailureMails")
	if ok {
		logins := strings.Split(failureMails, ",")
		for _, login := range logins {
			login = strings.TrimSpace(login)
			if len(login) == 0 {
				continue
			}
			result.FailureMails = append(result.FailureMails, login)
		}
	}
	return result, nil
}

func parseOptionValue(
	optionType models.OptionType, value string,
) (interface{}, error) {
	switch optionType {
	case models.StringOption:
		return value, nil
	case models.ConfigOption:
		// Config option is the same as integer option
		fallthrough
	case models.IntegerOption:
		number, err := strconv.ParseInt(value, 10, 64)
		return number, err
	case models.FloatOption:
		number, err := strconv.ParseFloat(value, 64)
		return number, err
	case models.SecretOption:
		var secret models.SecretValue
		err := json.Unmarshal([]byte(value), &secret)
		if err != nil {
			return strconv.ParseInt(value, 10, 64)
		}
		secret.Store = "Local"
		return secret, nil
	case models.JSONQuery:
		var query interface{}
		err := json.Unmarshal([]byte(value), &query)
		return query, err
	default:
		return nil, fmt.Errorf(
			"unsupported option type '%s'", optionType,
		)
	}
}

func getParamValue(values url.Values, name string) (string, bool) {
	value, ok := values[name]
	if !ok || len(value) != 1 {
		return "", false
	}
	return value[0], true
}

func parseTaskOptions(
	options models.ActionOptions, values url.Values,
) (models.TaskOptions, error) {
	result := models.TaskOptions{}
	expr, err := regexp.Compile(`^OptionValue\[(@?[a-zA-Z0-9\_]+)\]$`)
	if err != nil {
		return nil, err
	}
	for key, value := range values {
		if len(value) == 0 || !expr.MatchString(key) {
			continue
		}
		name := expr.FindStringSubmatch(key)[1]
		option, ok := options[name]
		if !ok {
			continue
		}
		opt := models.TaskOption{Type: option.Type}
		opt.Value, err = parseOptionValue(option.Type, value[0])
		if err != nil {
			return nil, err
		}
		result[name] = opt
	}
	return result, nil
}

func parseActionOption(
	name string, optionType models.OptionType, values url.Values,
) (opt models.ActionOption, err error) {
	opt.Type = optionType
	if value, ok := values["OptionTitle["+name+"]"]; ok {
		if len(value) > 0 {
			opt.Title = value[0]
		}
	}
	if value, ok := values["OptionDescription["+name+"]"]; ok {
		if len(value) > 0 {
			opt.Description = value[0]
		}
	}
	if value, ok := values["OptionRequired["+name+"]"]; ok {
		if len(value) > 0 {
			opt.Required = isChecked(value[0])
		}
	}
	if value, ok := values["OptionEditable["+name+"]"]; ok {
		if len(value) > 0 {
			opt.Editable = isChecked(value[0])
		}
	}
	if value, ok := values["OptionVisible["+name+"]"]; ok {
		if len(value) > 0 {
			opt.Visible = isChecked(value[0])
		}
	}
	if value, ok := values["OptionSettings["+name+"]"]; ok {
		if len(value) > 0 {
			err = json.Unmarshal([]byte(value[0]), &opt.Settings)
			if err != nil {
				return
			}
		}
	}
	if value, ok := values["OptionValue["+name+"]"]; ok {
		opt.Value, err = parseOptionValue(optionType, value[0])
	}
	return
}

func parseActionOptions(values url.Values) (models.ActionOptions, error) {
	options := models.ActionOptions{}
	expr, err := regexp.Compile(`^OptionType\[(@?[a-zA-Z0-9\_]+)\]$`)
	if err != nil {
		return nil, err
	}
	for key, value := range values {
		if len(value) == 0 || !expr.MatchString(key) {
			continue
		}
		name := expr.FindStringSubmatch(key)[1]
		optionType := models.OptionType(value[0])
		option, err := parseActionOption(name, optionType, values)
		if err != nil {
			return nil, err
		}
		options[name] = option
	}
	return options, nil
}

func getTemplateFuncs() template.FuncMap {
	return template.FuncMap{
		"ParseUnix": func(t int64) time.Time {
			return time.Unix(t, 0)
		},
		"Duration": func(begin, end int64) time.Duration {
			return time.Unix(end, 0).Sub(time.Unix(begin, 0))
		},
		"Now": func() int64 {
			return time.Now().Unix()
		},
		"Encode": func(data interface{}) string {
			bytes, err := json.MarshalIndent(data, "", "  ")
			if err != nil {
				return ""
			}
			return string(bytes)
		},
	}
}

func setupTemplates(cfg config.Server, e *echo.Echo) {
	tmpl := template.New("front").Funcs(getTemplateFuncs())
	dir := path.Join(cfg.OldTemplatesDir, "*.html")
	e.Renderer = &Template{
		templates: template.Must(tmpl.ParseGlob(dir)),
	}
}

func (v *View) ensureUserCreated(u *models.Account) error {
	if u.ID > 0 {
		return nil
	}
	return fmt.Errorf("not implemented")
}

func (v *View) getActionByContext(
	ctx echo.Context,
) (m models.Action, err error) {
	actionID, err := strconv.Atoi(ctx.Param("ActionID"))
	if err != nil {
		ctx.Logger().Error(err)
		return
	}
	m, err = v.app.Actions.Get(actionID)
	return
}

func (v *View) getPlannerByContext(
	ctx echo.Context,
) (m models.Planner, err error) {
	plannerID, err := strconv.Atoi(ctx.Param("PlannerID"))
	if err != nil {
		ctx.Logger().Error(err)
		return
	}
	m, err = v.app.Planners.Get(plannerID)
	return
}

func (v *View) getConfigByContext(
	ctx echo.Context,
) (m models.Config, err error) {
	configID, err := strconv.Atoi(ctx.Param("ConfigID"))
	if err != nil {
		ctx.Logger().Error(err)
		return
	}
	m, err = v.app.Configs.Get(configID)
	return
}
