package front

import (
	"database/sql"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/runner/models"
)

func (v *View) PostActionPlan(c echo.Context) error {
	params, err := c.FormParams()
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	options, err := parseTaskOptions(action.Options, params)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	settings, err := parsePlannerSettings(params)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	user := c.Get("auth_account").(models.Account)
	err = v.ensureUserCreated(&user)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	planner := models.Planner{
		ActionID:    int(action.ID),
		DirID:       int(action.DirID),
		OwnerID:     user.ID,
		Title:       c.FormValue("Title"),
		Description: c.FormValue("Description"),
		Settings:    settings,
		Options:     options,
	}
	if err := v.app.Planners.CreateTx(v.app.DB, &planner); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusFound,
		fmt.Sprintf("/planner/%d", planner.ID),
	)
}

func (v *View) GetPlannerInfo(c echo.Context) error {
	id, err := strconv.ParseInt(c.Param("PlannerID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	planner, err := v.app.Planners.Get(int(id))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	action, err := v.app.Actions.Get(planner.ActionID)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	for name, option := range planner.Options {
		opt, ok := action.Options[name]
		if ok {
			opt.Value = option.Value
			action.Options[name] = opt
		}
	}
	tasks, err := v.app.Tasks.FindByPlanner(planner.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Render(
		http.StatusOK, "planner.html",
		map[string]interface{}{
			"Planner": planner, "Action": action, "Tasks": tasks,
		},
	)
}

func (v *View) GetPlannerEdit(c echo.Context) error {
	id, err := strconv.ParseInt(c.Param("PlannerID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	planner, err := v.app.Planners.Get(int(id))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	action, err := v.app.Actions.Get(planner.ActionID)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	for name, option := range planner.Options {
		opt, ok := action.Options[name]
		if ok {
			opt.Value = option.Value
			action.Options[name] = opt
		}
	}
	return c.Render(
		http.StatusOK, "planner_edit.html",
		map[string]interface{}{
			"Planner": planner, "Action": action,
		},
	)
}

func (v *View) PostPlannerEdit(c echo.Context) error {
	id, err := strconv.ParseInt(c.Param("PlannerID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	planner, err := v.app.Planners.Get(int(id))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	action, err := v.app.Actions.Get(planner.ActionID)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	params, err := c.FormParams()
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	options, err := parseTaskOptions(action.Options, params)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	settings, err := parsePlannerSettings(params)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	settings.Enabled = planner.Settings.Enabled
	planner.Title = c.FormValue("Title")
	planner.Description = c.FormValue("Description")
	planner.Settings = settings
	planner.Options = options
	if err := v.app.Planners.UpdateTx(v.app.DB, planner); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusFound,
		fmt.Sprintf("/planner/%d", planner.ID),
	)
}

func (v *View) PostPlannerEnable(c echo.Context) error {
	id, err := strconv.ParseInt(c.Param("PlannerID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	planner, err := v.app.Planners.Get(int(id))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	user := c.Get("auth_account").(models.Account)
	err = v.ensureUserCreated(&user)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	planner.Settings.Enabled = true
	state := models.PlannerState{
		PlannerID: planner.ID,
		NextTime:  models.NInt64(planner.Next(time.Now().Unix())),
	}
	if err := gosql.WithTxContext(
		c.Request().Context(), v.app.DB, nil,
		func(tx *sql.Tx) error {
			if err := v.app.Planners.UpdateTx(tx, planner); err != nil {
				return err
			}
			if err := v.app.PlannerStates.UpsertTx(tx, state, "next_time"); err != nil {
				return err
			}
			return nil
		},
	); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusFound, fmt.Sprintf("/planner/%d", planner.ID),
	)
}

func (v *View) PostPlannerDisable(c echo.Context) error {
	id, err := strconv.ParseInt(c.Param("PlannerID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	planner, err := v.app.Planners.Get(int(id))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	user := c.Get("auth_account").(models.Account)
	err = v.ensureUserCreated(&user)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	planner.Settings.Enabled = false
	state := models.PlannerState{
		PlannerID: planner.ID,
		NextTime:  0,
	}
	if err := gosql.WithTxContext(
		c.Request().Context(), v.app.DB, nil,
		func(tx *sql.Tx) error {
			if err := v.app.Planners.UpdateTx(tx, planner); err != nil {
				return err
			}
			if err := v.app.PlannerStates.UpsertTx(tx, state, "next_time"); err != nil {
				return err
			}
			return nil
		},
	); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusFound, fmt.Sprintf("/planner/%d", planner.ID),
	)
}

func (v *View) GetPlannerTreeRoot(c echo.Context) error {
	id, err := strconv.ParseInt(c.Param("PlannerID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	planner, err := v.app.Planners.Get(int(id))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if planner.DirID == 0 {
		return c.NoContent(http.StatusNotFound)
	}
	c.SetParamNames("PlannerID", "DirID")
	c.SetParamValues(
		c.Param("PlannerID"),
		fmt.Sprintf("%d", planner.DirID),
	)
	return v.GetPlannerTreeList(c)
}

func (v *View) GetPlannerTreeList(c echo.Context) error {
	planner, err := v.getPlannerByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if planner.DirID == 0 {
		return c.NoContent(http.StatusNotFound)
	}
	dirID, err := strconv.ParseInt(c.Param("DirID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	dir, err := v.app.Nodes.Get(int(dirID))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	dirs, err := v.app.Nodes.FindByDir(int(dirID))
	if err != nil {
		c.Logger().Error(err)
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if dirs == nil {
		dirs = make([]models.Node, 0)
	}
	if dir.NodeID != 0 {
		parent, err := v.app.Nodes.Get(int(dir.NodeID))
		if err != nil {
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		return c.Render(
			http.StatusOK, "planner_tree.html",
			map[string]interface{}{
				"Parent": parent, "Dirs": dirs, "Dir": dir, "Planner": planner,
			},
		)
	}
	return c.Render(
		http.StatusOK, "planner_tree.html",
		map[string]interface{}{
			"Dirs": dirs, "Dir": dir, "Planner": planner,
		},
	)
}

func (v *View) PostPlannerTreeList(c echo.Context) error {
	planner, err := v.getPlannerByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	dirID, err := strconv.ParseInt(c.Param("DirID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	dir, err := v.app.Nodes.Get(int(dirID))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	planner.DirID = dir.ID
	if err := v.app.Planners.UpdateTx(v.app.DB, planner); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusFound,
		fmt.Sprintf("/planner/%d", planner.ID),
	)
}
