package front

import (
	"database/sql"
	"fmt"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/models"
)

func (v *View) PostActionRun(c echo.Context) error {
	params, err := c.FormParams()
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	action, err := v.getActionByContext(c)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	options, err := parseTaskOptions(action.Options, params)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	user := c.Get("auth_account").(models.Account)
	err = v.ensureUserCreated(&user)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	task := models.Task{
		ActionID: int(action.ID),
		OwnerID:  models.NInt(user.ID),
		Options:  options,
	}
	err = v.app.Tasks.Create(&task)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(http.StatusFound, fmt.Sprintf("/task/%d", task.ID))
}

func (v *View) GetTaskInfo(c echo.Context) error {
	id, err := strconv.ParseInt(c.Param("TaskID"), 10, 30)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusBadRequest)
	}
	task, err := v.app.Tasks.Get(int(id))
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	action, err := v.app.Actions.Get(task.ActionID)
	if err != nil {
		if err == sql.ErrNoRows {
			return c.NoContent(http.StatusNotFound)
		}
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	for name, option := range task.Options {
		opt, ok := action.Options[name]
		if ok {
			opt.Value = option.Value
			action.Options[name] = opt
		}
	}
	var planner models.Planner
	if task.PlannerID != 0 {
		planner, err = v.app.Planners.Get(int(task.PlannerID))
		if err != nil {
			if err == sql.ErrNoRows {
				return c.NoContent(http.StatusNotFound)
			}
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
	}
	logs, err := v.app.TaskLogs.FindByTask(task.ID)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Render(http.StatusOK, "task.html", struct {
		models.Task
		Logs    []models.TaskLog
		Action  models.Action
		Planner models.Planner
	}{Task: task, Logs: logs, Action: action, Planner: planner})
}

func (v *View) GetQueue(c echo.Context) error {
	return c.Render(http.StatusOK, "queue.html", nil)
}
